// SPDX-FileCopyrightText: 2022 Declan Rixon <twisted.cubing@gmail.com>
//
// SPDX-License-Identifier: GPL-3.0-only

use crate::cube::{Axis, Face};

pub fn face_containing_bar(bar: &'static [usize; 2]) -> Option<Face> {
    use bar::*;
    use Face::*;
    match bar {
        b if b == &UR_UL => Some(U),
        b if b == &UF_UB => Some(U),
        b if b == &DR_DL => Some(D),
        b if b == &DF_DB => Some(D),
        b if b == &RU_RD => Some(R),
        b if b == &RF_RB => Some(R),
        b if b == &LU_LD => Some(L),
        b if b == &LF_LB => Some(L),
        b if b == &FR_FL => Some(F),
        b if b == &FU_FD => Some(F),
        b if b == &BR_BL => Some(B),
        b if b == &BU_BD => Some(B),
        _ => None,
    }
}

pub fn corner_pair_between_bars(
    bar1: &'static [usize; 2],
    bar2: &'static [usize; 2],
) -> Option<&'static [usize; 2]> {
    use bar::*;
    use corner::adjacent_pair::*;
    match (bar1, bar2) {
        // U
        x if x == (&UR_UL, &BR_BL) || x == (&BR_BL, &UR_UL) => Some(&UBL_UBR),
        x if x == (&UF_UB, &LF_LB) || x == (&LF_LB, &UF_UB) => Some(&UBL_UFL),
        x if x == (&UF_UB, &RF_RB) || x == (&RF_RB, &UF_UB) => Some(&UBR_UFR),
        x if x == (&UR_UL, &FR_FL) || x == (&FR_FL, &UR_UL) => Some(&UFR_UFL),
        // U-D
        x if x == (&BU_BD, &LU_LD) || x == (&LU_LD, &BU_BD) => Some(&UBL_DBL),
        x if x == (&BU_BD, &RU_RD) || x == (&RU_RD, &BU_BD) => Some(&UBR_DBR),
        x if x == (&FU_FD, &RU_RD) || x == (&RU_RD, &FU_FD) => Some(&UFR_DFR),
        x if x == (&FU_FD, &LU_LD) || x == (&LU_LD, &FU_FD) => Some(&UFL_DFL),
        // D
        x if x == (&BR_BL, &DR_DL) || x == (&DR_DL, &BR_BL) => Some(&DBL_DBR),
        x if x == (&RF_RB, &DF_DB) || x == (&DF_DB, &RF_RB) => Some(&DBR_DFR),
        x if x == (&FR_FL, &DR_DL) || x == (&DR_DL, &FR_FL) => Some(&DFR_DFL),
        x if x == (&LF_LB, &DF_DB) || x == (&DF_DB, &LF_LB) => Some(&DFL_DBL),
        _ => None,
    }
}

pub fn faces_containing_adjacent_corner_pair(
    corner_pair: &'static [usize; 2],
) -> Option<[Face; 2]> {
    use corner::adjacent_pair::*;
    use Face::*;
    match corner_pair {
        x if x == &UBL_UBR => Some([U, B]),
        x if x == &UBL_UFL => Some([U, L]),
        x if x == &UBR_UFR => Some([U, R]),
        x if x == &UFR_UFL => Some([U, F]),
        x if x == &UBL_DBL => Some([B, L]),
        x if x == &UBR_DBR => Some([B, R]),
        x if x == &UFR_DFR => Some([F, R]),
        x if x == &UFL_DFL => Some([F, L]),
        x if x == &DBL_DBR => Some([D, B]),
        x if x == &DBR_DFR => Some([D, R]),
        x if x == &DFR_DFL => Some([D, F]),
        x if x == &DFL_DBL => Some([D, L]),
        _ => None,
    }
}

pub fn eo_bars_by_axes(
    eo_axis: Axis,
    slice_axis: Axis,
) -> Option<&'static [&'static [usize; 2]; 2]> {
    use bar::eo_bars_by_axis::*;
    match (eo_axis, slice_axis) {
        (Axis::X, Axis::Y) => Some(&XY),
        (Axis::X, Axis::Z) => Some(&XZ),
        (Axis::Y, Axis::X) => Some(&YX),
        (Axis::Y, Axis::Z) => Some(&YZ),
        (Axis::Z, Axis::X) => Some(&ZX),
        (Axis::Z, Axis::Y) => Some(&ZY),
        _ => None,
    }
}

mod bar {
    pub static UR_UL: [usize; 2] = [1, 3];
    pub static UF_UB: [usize; 2] = [2, 0];
    pub static DR_DL: [usize; 2] = [9, 11];
    pub static DF_DB: [usize; 2] = [8, 10];
    pub static RU_RD: [usize; 2] = [1, 9];
    pub static RF_RB: [usize; 2] = [6, 5];
    pub static LU_LD: [usize; 2] = [3, 11];
    pub static LF_LB: [usize; 2] = [7, 4];
    pub static FR_FL: [usize; 2] = [6, 7];
    pub static FU_FD: [usize; 2] = [2, 10];
    pub static BR_BL: [usize; 2] = [5, 4];
    pub static BU_BD: [usize; 2] = [0, 8];
    pub mod eo_bars_by_axis {
        use super::*;
        pub static XY: [&[usize; 2]; 2] = [&FR_FL, &BR_BL];
        pub static XZ: [&[usize; 2]; 2] = [&UR_UL, &DR_DL];
        pub static YX: [&[usize; 2]; 2] = [&FU_FD, &BU_BD];
        pub static YZ: [&[usize; 2]; 2] = [&LU_LD, &RU_RD];
        pub static ZX: [&[usize; 2]; 2] = [&UF_UB, &DF_DB];
        pub static ZY: [&[usize; 2]; 2] = [&LF_LB, &RF_RB];
    }
}

mod corner {
    pub mod adjacent_pair {
        // U
        pub static UBL_UBR: [usize; 2] = [0, 1];
        pub static UBL_UFL: [usize; 2] = [0, 3];
        pub static UBR_UFR: [usize; 2] = [1, 2];
        pub static UFR_UFL: [usize; 2] = [2, 3];
        // U-D
        pub static UBL_DBL: [usize; 2] = [0, 4];
        pub static UBR_DBR: [usize; 2] = [1, 5];
        pub static UFR_DFR: [usize; 2] = [2, 6];
        pub static UFL_DFL: [usize; 2] = [3, 7];
        // D
        pub static DBL_DBR: [usize; 2] = [4, 5];
        pub static DBR_DFR: [usize; 2] = [5, 6];
        pub static DFR_DFL: [usize; 2] = [6, 7];
        pub static DFL_DBL: [usize; 2] = [7, 4];
    }
}
