use crate::{
    guild::member::{Member, OptionalMemberDeserializer},
    id::{ChannelId, GuildId, UserId},
};
use serde::{
    de::{Deserializer, Error as DeError, IgnoredAny, MapAccess, Visitor},
    Deserialize, Serialize,
};
use std::fmt::{Formatter, Result as FmtResult};

#[derive(Clone, Debug, Eq, Hash, PartialEq, Serialize)]
pub struct TypingStart {
    pub channel_id: ChannelId,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub guild_id: Option<GuildId>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub member: Option<Member>,
    pub timestamp: u64,
    pub user_id: UserId,
}

#[derive(Debug, Deserialize)]
#[serde(field_identifier, rename_all = "snake_case")]
enum Field {
    ChannelId,
    GuildId,
    Member,
    Timestamp,
    UserId,
}

struct TypingStartVisitor;

impl<'de> Visitor<'de> for TypingStartVisitor {
    type Value = TypingStart;

    fn expecting(&self, f: &mut Formatter<'_>) -> FmtResult {
        f.write_str("struct TypingStart")
    }

    fn visit_map<V: MapAccess<'de>>(self, mut map: V) -> Result<Self::Value, V::Error> {
        let mut channel_id = None;
        let mut guild_id = None::<Option<_>>;
        let mut member = None::<Member>;
        let mut timestamp = None;
        let mut user_id = None;

        let span = tracing::trace_span!("deserializing typing start");
        let _span_enter = span.enter();

        loop {
            let span_child = tracing::trace_span!("iterating over element");
            let _span_child_enter = span_child.enter();

            let key = match map.next_key() {
                Ok(Some(key)) => {
                    tracing::trace!(?key, "found key");

                    key
                }
                Ok(None) => break,
                Err(why) => {
                    // Encountered when we run into an unknown key.
                    map.next_value::<IgnoredAny>()?;

                    tracing::trace!("ran into an unknown key: {:?}", why);

                    continue;
                }
            };

            match key {
                Field::ChannelId => {
                    if channel_id.is_some() {
                        return Err(DeError::duplicate_field("channel_id"));
                    }

                    channel_id = Some(map.next_value()?);
                }
                Field::GuildId => {
                    if guild_id.is_some() {
                        return Err(DeError::duplicate_field("guild_id"));
                    }

                    guild_id = Some(map.next_value()?);
                }
                Field::Member => {
                    if member.is_some() {
                        return Err(DeError::duplicate_field("member"));
                    }

                    let deserializer =
                        OptionalMemberDeserializer::new(GuildId::new(1).expect("non zero"));

                    member = map.next_value_seed(deserializer)?;
                }
                Field::Timestamp => {
                    if timestamp.is_some() {
                        return Err(DeError::duplicate_field("timestamp"));
                    }

                    timestamp = Some(map.next_value()?);
                }
                Field::UserId => {
                    if user_id.is_some() {
                        return Err(DeError::duplicate_field("user_id"));
                    }

                    user_id = Some(map.next_value()?);
                }
            }
        }

        let channel_id = channel_id.ok_or_else(|| DeError::missing_field("channel_id"))?;
        let guild_id = guild_id.unwrap_or_default();
        let timestamp = timestamp.ok_or_else(|| DeError::missing_field("timestamp"))?;
        let user_id = user_id.ok_or_else(|| DeError::missing_field("user_id"))?;

        tracing::trace!(
            %channel_id,
            ?guild_id,
            %timestamp,
            %user_id,
        );

        if let (Some(guild_id), Some(member)) = (guild_id, member.as_mut()) {
            tracing::trace!(%guild_id, ?member, "setting member guild id");

            member.guild_id = guild_id;
        }

        Ok(TypingStart {
            channel_id,
            guild_id,
            member,
            timestamp,
            user_id,
        })
    }
}

impl<'de> Deserialize<'de> for TypingStart {
    fn deserialize<D: Deserializer<'de>>(deserializer: D) -> Result<Self, D::Error> {
        const FIELDS: &[&str] = &["channel_id", "guild_id", "member", "timestamp", "user_id"];

        deserializer.deserialize_struct("TypingStart", FIELDS, TypingStartVisitor)
    }
}

#[cfg(test)]
mod tests {
    use super::TypingStart;
    use crate::{
        datetime::{Timestamp, TimestampParseError},
        guild::Member,
        id::{ChannelId, GuildId, RoleId, UserId},
        user::User,
    };
    use serde_test::Token;
    use std::str::FromStr;

    #[allow(clippy::too_many_lines)]
    #[test]
    fn test_typing_start_with_member() -> Result<(), TimestampParseError> {
        let joined_at = Timestamp::from_str("2020-01-01T00:00:00.000000+00:00")?;

        let value = TypingStart {
            channel_id: ChannelId::new(2).expect("non zero"),
            guild_id: Some(GuildId::new(1).expect("non zero")),
            member: Some(Member {
                deaf: false,
                guild_id: GuildId::new(1).expect("non zero"),
                joined_at: Some(joined_at),
                mute: false,
                nick: Some("typing".to_owned()),
                pending: false,
                premium_since: None,
                roles: vec![RoleId::new(4).expect("non zero")],
                user: User {
                    id: UserId::new(3).expect("non zero"),
                    accent_color: None,
                    avatar: Some("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa".to_owned()),
                    banner: None,
                    bot: false,
                    discriminator: 1,
                    name: "test".to_owned(),
                    mfa_enabled: None,
                    locale: None,
                    verified: None,
                    email: None,
                    flags: None,
                    premium_type: None,
                    system: None,
                    public_flags: None,
                },
            }),
            timestamp: 1_500_000_000,
            user_id: UserId::new(3).expect("non zero"),
        };

        serde_test::assert_tokens(
            &value,
            &[
                Token::Struct {
                    name: "TypingStart",
                    len: 5,
                },
                Token::Str("channel_id"),
                Token::NewtypeStruct { name: "ChannelId" },
                Token::Str("2"),
                Token::Str("guild_id"),
                Token::Some,
                Token::NewtypeStruct { name: "GuildId" },
                Token::Str("1"),
                Token::Str("member"),
                Token::Some,
                Token::Struct {
                    name: "Member",
                    len: 8,
                },
                Token::Str("deaf"),
                Token::Bool(false),
                Token::Str("guild_id"),
                Token::NewtypeStruct { name: "GuildId" },
                Token::Str("1"),
                Token::Str("joined_at"),
                Token::Some,
                Token::Str("2020-01-01T00:00:00.000000+00:00"),
                Token::Str("mute"),
                Token::Bool(false),
                Token::Str("nick"),
                Token::Some,
                Token::Str("typing"),
                Token::Str("pending"),
                Token::Bool(false),
                Token::Str("roles"),
                Token::Seq { len: Some(1) },
                Token::NewtypeStruct { name: "RoleId" },
                Token::Str("4"),
                Token::SeqEnd,
                Token::Str("user"),
                Token::Struct {
                    name: "User",
                    len: 7,
                },
                Token::Str("accent_color"),
                Token::None,
                Token::Str("avatar"),
                Token::Some,
                Token::Str("aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"),
                Token::Str("banner"),
                Token::None,
                Token::Str("bot"),
                Token::Bool(false),
                Token::Str("discriminator"),
                Token::Str("0001"),
                Token::Str("id"),
                Token::NewtypeStruct { name: "UserId" },
                Token::Str("3"),
                Token::Str("username"),
                Token::Str("test"),
                Token::StructEnd,
                Token::StructEnd,
                Token::Str("timestamp"),
                Token::U64(1_500_000_000),
                Token::Str("user_id"),
                Token::NewtypeStruct { name: "UserId" },
                Token::Str("3"),
                Token::StructEnd,
            ],
        );

        Ok(())
    }

    #[test]
    fn test_typing_start_without_member() {
        let value = TypingStart {
            channel_id: ChannelId::new(2).expect("non zero"),
            guild_id: None,
            member: None,
            timestamp: 1_500_000_000,
            user_id: UserId::new(3).expect("non zero"),
        };

        serde_test::assert_tokens(
            &value,
            &[
                Token::Struct {
                    name: "TypingStart",
                    len: 3,
                },
                Token::Str("channel_id"),
                Token::NewtypeStruct { name: "ChannelId" },
                Token::Str("2"),
                Token::Str("timestamp"),
                Token::U64(1_500_000_000),
                Token::Str("user_id"),
                Token::NewtypeStruct { name: "UserId" },
                Token::Str("3"),
                Token::StructEnd,
            ],
        );
    }
}
