use thiserror::Error;
use twilight_model::application::interaction::application_command::CommandDataOption;

/// The type of the errors returned by the functions in `CommandDataOptionsTrait`
#[derive(Error, Debug)]
pub enum CommandDataOptionsConversionError {
    /// Returned if the `Vec` doesn't have `index` elements
    #[error("Vec doesn't have enough elements")]
    NotEnoughItemsInVec,
    /// Returned if `string` is called on a `Vec<CommandDataOption>` whose item at `index` isn't `String` variant
    #[error("Command data option is not string")]
    NotString,
    /// Returned if `integer` is called on a `Vec<CommandDataOption>` whose item at `index` isn't `Integer` variant
    #[error("Command data option is not integer")]
    NotInteger,
    /// Returned if `boolean` is called on a `Vec<CommandDataOption>` whose item at `index` isn't `Boolean` variant
    #[error("Command data option is not boolean")]
    NotBoolean,
    /// Returned if `subcommand` is called on a `Vec<CommandDataOption>` whose first item is not `SubCommand` variant
    #[error("Command data option is not subcommand")]
    NotSubCommand,
}

/// Returned from the `string` method in `CommandDataOptionsTrait`
#[derive(Debug)]
pub struct CommandDataOptionString {
    pub name: String,
    pub value: String,
}

/// Returned from the `integer` method in `CommandDataOptionsTrait`
#[derive(Debug)]
pub struct CommandDataOptionInteger {
    pub name: String,
    pub value: i64,
}

/// Returned from the `boolean` method in `CommandDataOptionsTrait`
#[derive(Debug)]
pub struct CommandDataOptionBoolean {
    pub name: String,
    pub value: bool,
}

/// Returned from the `subcommand` method in `CommandDataOptionsTrait`
#[derive(Debug)]
pub struct CommandDataOptionSubCommand {
    pub name: String,
    pub options: Vec<CommandDataOption>,
}

/// Trait to get a `String`, `Integer`, `Boolean` or `SubCommand` from `CommandDataOption`
pub trait CommandDataOptionsTrait {
    /// Returns the wrapped `CommandDataOptionString` if `CommandDataOption` is the `String` variant
    fn string(self) -> Result<CommandDataOptionString, CommandDataOptionsConversionError>;
    /// Returns the wrapped `CommandDataOptionInteger` if `CommandDataOption` is the `Integer` variant
    fn integer(self) -> Result<CommandDataOptionInteger, CommandDataOptionsConversionError>;
    /// Returns the wrapped `CommandDataOptionBoolean` if `CommandDataOption` is the `Boolean` variant
    fn boolean(self) -> Result<CommandDataOptionBoolean, CommandDataOptionsConversionError>;
    /// Returns the wrapped `CommandDataOptionSubCommand` if `CommandDataOption` is the `SubCommand` variant
    fn subcommand(self) -> Result<CommandDataOptionSubCommand, CommandDataOptionsConversionError>;
}

impl CommandDataOptionsTrait for CommandDataOption {
    fn string(self) -> Result<CommandDataOptionString, CommandDataOptionsConversionError> {
        if let CommandDataOption::String { name, value } = self {
            Ok(CommandDataOptionString { name, value })
        } else {
            Err(CommandDataOptionsConversionError::NotString)
        }
    }

    fn integer(self) -> Result<CommandDataOptionInteger, CommandDataOptionsConversionError> {
        if let CommandDataOption::Integer { name, value } = self {
            Ok(CommandDataOptionInteger { name, value })
        } else {
            Err(CommandDataOptionsConversionError::NotInteger)
        }
    }

    fn boolean(self) -> Result<CommandDataOptionBoolean, CommandDataOptionsConversionError> {
        if let CommandDataOption::Boolean { name, value } = self {
            Ok(CommandDataOptionBoolean { name, value })
        } else {
            Err(CommandDataOptionsConversionError::NotBoolean)
        }
    }

    fn subcommand(self) -> Result<CommandDataOptionSubCommand, CommandDataOptionsConversionError> {
        if let CommandDataOption::SubCommand { name, options } = self {
            Ok(CommandDataOptionSubCommand { name, options })
        } else {
            Err(CommandDataOptionsConversionError::NotSubCommand)
        }
    }
}
