use thiserror::Error;
use twilight_model::application::interaction::{ApplicationCommand, Interaction, Ping};

/// The type of the errors returned by the functions in `InteractionTrait`
#[derive(Error, Debug)]
pub enum InteractionConversionError {
    /// Returned if `ping` is called on an `Interaction` whose variant isn't `Ping`
    #[error("Interaction is not ping")]
    NotPing,
    /// Returned if `application_command` is called on an `Interaction` whose variant isn't `ApplicationCommand`
    #[error("Interaction is not application command")]
    NotApplicationCommand,
}

/// Trait to get the `Ping` or `ApplicationCommand` from an `Interaction`
pub trait InteractionTrait {
    fn ping(self) -> Result<Ping, InteractionConversionError>;
    fn application_command(self) -> Result<ApplicationCommand, InteractionConversionError>;
}

impl InteractionTrait for Interaction {
    /// Returns the wrapped `Ping` if `Interaction` is the `Ping` variant
    fn ping(self) -> Result<Ping, InteractionConversionError> {
        if let Interaction::Ping(ping) = self {
            Ok(*ping)
        } else {
            Err(InteractionConversionError::NotPing)
        }
    }

    /// Returns the wrapped `ApplicationCommand` if `Interaction` is the `ApplicationCommand` variant
    fn application_command(self) -> Result<ApplicationCommand, InteractionConversionError> {
        if let Interaction::ApplicationCommand(command) = self {
            Ok(*command)
        } else {
            Err(InteractionConversionError::NotApplicationCommand)
        }
    }
}
