use thiserror::Error;
use twilight_model::application::interaction::application_command::CommandDataOption;

/// The type of the errors returned by the functions in `CommandDataOptionsTrait`
#[derive(Error, Debug)]
pub enum CommandDataOptionsConversionError {
    /// Returned if the `Vec` doesn't have `index` elements
    #[error("Vec doesn't have enough elements")]
    NotEnoughItemsInVec,
    /// Returned if `string` is called on a `Vec<CommandDataOption>` whose item at `index` isn't `String` variant
    #[error("Command data option is not string")]
    NotString,
    /// Returned if `integer` is called on a `Vec<CommandDataOption>` whose item at `index` isn't `Integer` variant
    #[error("Command data option is not integer")]
    NotInteger,
    /// Returned if `boolean` is called on a `Vec<CommandDataOption>` whose item at `index` isn't `Boolean` variant
    #[error("Command data option is not boolean")]
    NotBoolean,
    /// Returned if `subcommand` is called on a `Vec<CommandDataOption>` whose first item is not `SubCommand` variant
    #[error("Command data option is not subcommand")]
    NotSubCommand,
}

/// Returned from the `string` method in `CommandDataOptionsTrait`
pub struct CommandDataOptionString {
    pub name: String,
    pub value: String,
}

/// Returned from the `integer` method in `CommandDataOptionsTrait`
pub struct CommandDataOptionInteger {
    pub name: String,
    pub value: i64,
}

/// Returned from the `boolean` method in `CommandDataOptionsTrait`
pub struct CommandDataOptionBoolean {
    pub name: String,
    pub value: bool,
}

/// Returned from the `subcommand` method in `CommandDataOptionsTrait`
pub struct CommandDataOptionSubCommand {
    pub name: String,
    pub options: Vec<CommandDataOption>,
}

/// Trait to get a `String`, `Integer`, `Boolean` or `SubCommand` from `Vec<CommandDataOption>`
///
/// Note that this trait removes the value from the `Vec` to avoid clones
pub trait CommandDataOptionsTrait {
    fn string(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOptionString, CommandDataOptionsConversionError>;
    fn integer(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOptionInteger, CommandDataOptionsConversionError>;
    fn boolean(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOptionBoolean, CommandDataOptionsConversionError>;
    fn subcommand(
        &mut self,
    ) -> Result<CommandDataOptionSubCommand, CommandDataOptionsConversionError>;
}

impl CommandDataOptionsTrait for Vec<CommandDataOption> {
    /// Removes the value at `index` and returns the wrapped `CommandDataOptionString` if `CommandDataOption` is the `String` variant
    fn string(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOptionString, CommandDataOptionsConversionError> {
        if let CommandDataOption::String { name, value } = self.index(index)? {
            Ok(CommandDataOptionString { name, value })
        } else {
            Err(CommandDataOptionsConversionError::NotString)
        }
    }

    /// Removes the value at `index` and returns the wrapped `CommandDataOptionInteger` if `CommandDataOption` is the `Integer` variant
    fn integer(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOptionInteger, CommandDataOptionsConversionError> {
        if let CommandDataOption::Integer { name, value } = self.index(index)? {
            Ok(CommandDataOptionInteger { name, value })
        } else {
            Err(CommandDataOptionsConversionError::NotInteger)
        }
    }

    /// Removes the value at `index` and returns the wrapped `CommandDataOptionBoolean` if `CommandDataOption` is the `Boolean` variant
    fn boolean(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOptionBoolean, CommandDataOptionsConversionError> {
        if let CommandDataOption::Boolean { name, value } = self.index(index)? {
            Ok(CommandDataOptionBoolean { name, value })
        } else {
            Err(CommandDataOptionsConversionError::NotBoolean)
        }
    }

    /// Removes the first value in the `Vec` and returns the wrapped `CommandDataOptionSubCommand` if `CommandDataOption` is the `SubCommand` variant
    fn subcommand(
        &mut self,
    ) -> Result<CommandDataOptionSubCommand, CommandDataOptionsConversionError> {
        if let CommandDataOption::SubCommand { name, options } = self.index(0)? {
            Ok(CommandDataOptionSubCommand { name, options })
        } else {
            Err(CommandDataOptionsConversionError::NotSubCommand)
        }
    }
}

trait Index {
    fn index(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOption, CommandDataOptionsConversionError>;
}

impl Index for Vec<CommandDataOption> {
    fn index(
        &mut self,
        index: usize,
    ) -> Result<CommandDataOption, CommandDataOptionsConversionError> {
        if self.len() < index + 1 {
            Err(CommandDataOptionsConversionError::NotEnoughItemsInVec)
        } else {
            Ok(self.remove(index))
        }
    }
}
