use std::fs;
use std::io::{self, Read};
use std::path::Path;

fn slurp<P: AsRef<Path>>(path: P) -> io::Result<Vec<u8>> {
    let mut f = fs::File::open(path)?;
    let mut buf = vec![];
    f.read_to_end(&mut buf)?;
    Ok(buf)
}

macro_rules! test {
    ( $name:ident $( , $args:expr )* ) => {
        #[test]
        fn $name() {
            use std::fs;
            use std::process::Command;
            use colored::Colorize;
            use diff;
            use crate::slurp;

            let output = Command::new("cargo")
                .arg("run")
                .arg("--")
                $(
                    .arg($args)
                )*
                .current_dir(concat!(env!("CARGO_MANIFEST_DIR"), "/tests/all/"))
                .output()
                .unwrap();

            assert!(
                output.status.success(),
                "should have run `twiggy` OK\n\n\
                 ============================== stdout ==============================\n\n\
                 {}\n\n\
                 ============================== stderr ==============================\n\n\
                 {}\n\n",
                String::from_utf8_lossy(&output.stdout),
                String::from_utf8_lossy(&output.stderr),
            );

            let expected_path = concat!(
                env!("CARGO_MANIFEST_DIR"),
                "/tests/all/expectations/",
                stringify!($name)
            );

            // Ignore errors. The diffing will provide a better diagnostic report.
            let expected = slurp(expected_path).unwrap_or(vec![]);
            let expected = String::from_utf8_lossy(&expected);
            let expected_lines = expected.lines().collect::<Vec<&str>>();

            let actual = String::from_utf8_lossy(&output.stdout);

            if ::std::env::var("TWIGGY_UPDATE_TEST_EXPECTATIONS").is_ok() {
                fs::write(expected_path, actual.as_ref()).unwrap();
                return;
            }

            let actual_lines = actual.lines().collect::<Vec<&str>>();

            if actual_lines != expected_lines {
                let mut cmd = "twiggy".to_string();
                $(
                    cmd.push(' ');
                    cmd.push_str($args);
                )*
                println!(
                    "\n{} {}\n",
                    format!("`{}`", cmd).red(),
                    "did not have the expected output!".red()
                );
                println!("--- {}", expected_path);
                println!(
                    "{} {}\n",
                    "+++ actually generated by".red(),
                    format!("`{}`", cmd).red()
                );
                for diff in diff::slice(&expected_lines, &actual_lines) {
                    match diff {
                        diff::Result::Left(l) => {
                            println!("{}", format!("-{}", l).red())
                        }
                        diff::Result::Both(l, _) => println!(" {}", l),
                        diff::Result::Right(r) => {
                            println!("{}", format!("+{}", r).red())
                        }
                    }
                }
                println!();
                panic!();
            }
        }
    }
}

mod diff_tests;
mod dominators_tests;
mod elf_format_tests;
mod garbage_tests;
mod monos_tests;
mod paths_tests;
mod top_tests;
