use std::{collections::HashMap, fmt};

use camino::Utf8PathBuf;
use serde::{Deserialize, Serialize};
use subprocess::{Exec, Redirection};
use thiserror::Error as ThisError;

use super::{command, file, Status};

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(rename_all = "lowercase")]
pub enum DesiredState {
    Absent,
    Latest,
    Present,
}

// TODO: wait for https://github.com/rust-lang/rust/issues/35121
// TODO: drop this and use the std::never::Never type instead
#[derive(Debug, ThisError)]
pub enum Error {
    #[error(transparent)]
    CommandJob {
        #[from]
        source: command::Error,
    },
    #[error(transparent)]
    FileJob {
        #[from]
        source: file::Error,
    },
    #[allow(dead_code)]
    #[error("never")]
    Never,
}
impl PartialEq for Error {
    fn eq(&self, other: &Error) -> bool {
        format!("{:?}", self) == format!("{:?}", other)
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(default, rename_all = "lowercase", tag = "type")]
pub struct InstallerNpm {
    /// Packages to target?
    pub packages: Vec<String>,
    /// Custom path for local mode, otherwise run against global prefix (default), see [`npm-prefix`](https://docs.npmjs.com/cli/v7/commands/npm-prefix)
    pub local: Option<Utf8PathBuf>,
    /// Action to perform?
    /// - [`Absent`](DesiredDate::Absent) to uninstall target [`packages`](InstallerNpm::packages)
    /// - [`Latest`](DesiredDate::Latest) to update all currently-installed packages
    /// - [`Present`](DesiredDate::Present) to install target [`packages`](InstallerNpm::packages)
    pub state: DesiredState,
}
impl Default for InstallerNpm {
    fn default() -> Self {
        Self {
            local: None,
            packages: vec![],
            state: DesiredState::Latest,
        }
    }
}
impl fmt::Display for InstallerNpm {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            "npm install {} {:?} packages",
            &self.packages.len(),
            &self.state
        )
    }
}
impl InstallerNpm {
    pub fn execute(&self) -> Result {
        match self.state {
            DesiredState::Absent => {
                if self.packages.is_empty() {
                    return Ok(Status::NoChange(String::from("0 packages to uninstall")));
                }

                let found: Vec<String> = self.found_versions().keys().map(String::from).collect();
                let surplus: Vec<String> = self
                    .packages
                    .iter()
                    .filter(|c| !c.trim().is_empty() && found.contains(c))
                    .map(String::from)
                    .collect();
                if surplus.is_empty() {
                    return Ok(Status::NoChange(String::from("0 packages uninstalled")));
                }

                let before = format!(
                    "{} of {} packages to install",
                    surplus.len(),
                    self.packages.len()
                );

                let mut cmd = self.npm_command();
                // unwrap: safe here because we're always Some from `self.npm_command()`
                cmd.argv.as_mut().unwrap().push(String::from("uninstall"));
                cmd.argv.as_mut().unwrap().extend(surplus);
                cmd.execute()?;

                Ok(Status::Changed(
                    before,
                    format!("{} packages uninstalled", self.packages.len()),
                ))
            }
            DesiredState::Latest => {
                let found_before = self.found_versions();

                let before = format!("{} packages", found_before.len());

                let mut cmd = self.npm_command();
                // unwrap: safe here because we're always Some from `self.npm_command()`
                cmd.argv.as_mut().unwrap().push(String::from("update"));
                cmd.execute()?;

                let found_after = self.found_versions();
                let found_changed = found_after
                    .into_iter()
                    .filter(|(package, version)| found_before.get(package) == Some(version))
                    .count();

                Ok(if found_changed == 0 {
                    Status::NoChange(before)
                } else {
                    Status::Changed(
                        before,
                        format!(
                            "{} of {} packages updated",
                            found_changed,
                            self.packages.len()
                        ),
                    )
                })
            }
            DesiredState::Present => {
                if self.packages.is_empty() {
                    return Ok(Status::NoChange(String::from("0 packages to install")));
                }

                let found: Vec<String> = self.found_versions().keys().map(String::from).collect();
                let missing: Vec<String> = self
                    .packages
                    .iter()
                    .filter(|c| !c.trim().is_empty() && !found.contains(c))
                    .map(String::from)
                    .collect();
                if missing.is_empty() {
                    return Ok(Status::NoChange(format!(
                        "{} packages installed",
                        self.packages.len()
                    )));
                }

                let before = format!(
                    "{} of {} packages to install",
                    missing.len(),
                    self.packages.len()
                );

                let mut cmd = self.npm_command();
                // unwrap: safe here because we're always Some from `self.npm_command()`
                cmd.argv.as_mut().unwrap().push(String::from("install"));
                cmd.argv.as_mut().unwrap().extend(missing);
                cmd.execute()?;

                Ok(Status::Changed(
                    before,
                    format!("{} packages installed", self.packages.len()),
                ))
            }
        }
    }

    fn found_versions(&self) -> HashMap<String, String> {
        let out = match self
            .npm_exec()
            .args(&["ls", "--depth=0", "--json"])
            .capture()
        {
            Ok(o) => o.stdout_str(),
            Err(_) => String::from(""),
        };
        let ls: NpmLs = match serde_json::from_str(&out) {
            Ok(ls) => ls,
            Err(e) => {
                eprintln!("error: unable to parse JSON from `npm ls`: {:?}", e);
                return HashMap::new();
            }
        };
        ls.dependencies
            .iter()
            .map(|(name, dep)| (name.clone(), dep.version.clone()))
            .collect()
    }

    fn npm_command(&self) -> command::Command {
        let mut cmd = command::Command {
            argv: Some(vec![]),
            command: String::from("npm"),
            ..Default::default()
        };
        match &self.local {
            Some(l) => cmd.chdir = Some(l.clone().into_std_path_buf()),
            // unwrap: safe here because we're always Some above
            None => cmd.argv.as_mut().unwrap().push(String::from("--global")),
        };
        cmd
    }

    fn npm_exec(&self) -> Exec {
        let exec = Exec::cmd("npm")
            .stdout(Redirection::Pipe)
            .stderr(Redirection::Merge);
        match &self.local {
            Some(l) => exec.cwd(l),
            None => exec.arg("--global"),
        }
    }
}

pub type Result = std::result::Result<Status, Error>;

#[derive(Debug, Deserialize)]
struct Dependency {
    #[serde(default)]
    version: String,
}

#[derive(Debug, Deserialize)]
struct NpmLs {
    #[serde(default)]
    dependencies: HashMap<String, Dependency>,
}

#[cfg(test)]
mod tests {
    use std::fs::metadata;

    use super::super::file::tests::temp_dir;

    use super::*;

    const SAMPLE_PACKAGE: &str = "open-cli";

    #[test]
    fn present_then_absent() -> std::result::Result<(), Error> {
        let tmp = temp_dir()?;
        let tmp_root = Utf8PathBuf::from_path_buf(tmp.to_path_buf())
            .expect("unable to convert temporary PathBuf");
        let tmp_bin = tmp_root.join("node_modules/.bin");
        assert!(metadata(&tmp_bin.join(SAMPLE_PACKAGE)).is_err());

        let present = InstallerNpm {
            local: Some(tmp_root.clone()),
            packages: vec![String::from(SAMPLE_PACKAGE)],
            state: DesiredState::Present,
        };

        let initial_versions = present.found_versions();
        assert_eq!(initial_versions.len(), 0);

        present.execute()?;

        assert!(metadata(&tmp_bin.join(SAMPLE_PACKAGE)).is_ok());

        let installed_versions = present.found_versions();
        assert_ne!(installed_versions.len(), 0);
        assert!(installed_versions.contains_key(SAMPLE_PACKAGE));

        let absent = InstallerNpm {
            local: Some(tmp_root),
            packages: vec![String::from(SAMPLE_PACKAGE)],
            state: DesiredState::Absent,
        };
        absent.execute()?;

        assert!(metadata(&tmp_bin.join(SAMPLE_PACKAGE)).is_err());

        let absent_versions = absent.found_versions();
        assert_eq!(absent_versions.len(), 0);

        Ok(())
    }
}
