use std::{collections::HashMap, fmt, path::PathBuf};

use semver::{Version, VersionReq};
use serde::{Deserialize, Serialize};
use subprocess::{Exec, Redirection};
use thiserror::Error as ThisError;

use super::{command, file, Status};

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(rename_all = "lowercase")]
pub enum DesiredState {
    Absent,
    Latest,
}

// TODO: wait for https://github.com/rust-lang/rust/issues/35121
// TODO: drop this and use the std::never::Never type instead
#[derive(Debug, ThisError)]
pub enum Error {
    #[error(transparent)]
    CommandJob {
        #[from]
        source: command::Error,
    },
    #[error(transparent)]
    FileJob {
        #[from]
        source: file::Error,
    },
    #[allow(dead_code)]
    #[error("never")]
    Never,
}
impl PartialEq for Error {
    fn eq(&self, other: &Error) -> bool {
        format!("{:?}", self) == format!("{:?}", other)
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(default, rename_all = "lowercase", tag = "type")]
pub struct InstallerGo {
    /// Set the GOBIN environment variable for `go`,
    /// where `go install` will put executable files
    pub gobin: Option<PathBuf>,
    /// Set the GOPATH environment variable for `go`,
    /// where `go `install` will put package source code
    pub gopath: Option<PathBuf>,
    pub packages: Vec<String>,
    pub state: DesiredState,
}
impl Default for InstallerGo {
    fn default() -> Self {
        Self {
            gobin: None,
            gopath: None,
            packages: vec![],
            state: DesiredState::Latest,
        }
    }
}
impl fmt::Display for InstallerGo {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            "go install {} {:?} packages",
            &self.packages.len(),
            &self.state
        )
    }
}
impl InstallerGo {
    pub fn execute(&self) -> Result {
        match self.state {
            DesiredState::Absent => {
                let gobin = self.gobin_dir();
                for pkg in &self.packages {
                    let exe = executable_name(&pkg);
                    let f = file::File {
                        force: Some(true),
                        path: gobin.join(exe),
                        src: None,
                        state: file::FileState::Absent,
                    };
                    f.execute()?;
                    // TODO: remove related directories from GOPATH/pkg/mod and GOPATH/src
                }
            }
            DesiredState::Latest => {
                let mut env: HashMap<String, String> = HashMap::new();
                if let Some(gobin) = &self.gobin {
                    env.insert(String::from("GOBIN"), gobin.to_string_lossy().into_owned());
                }
                if let Some(gopath) = &self.gopath {
                    env.insert(
                        String::from("GOPATH"),
                        gopath.to_string_lossy().into_owned(),
                    );
                }
                for pkg in &self.packages {
                    let target = if pkg.contains('@') {
                        pkg.clone()
                    } else {
                        format!("{}@latest", pkg)
                    };
                    let cmd = command::Command {
                        argv: Some(vec![String::from("install"), target]),
                        env: Some(env.clone()), // TODO: work out a non-clone solution
                        command: String::from("go"),
                        ..Default::default()
                    };
                    cmd.execute()?;
                }
            }
        }
        Ok(Status::Done)
    }

    fn gobin_dir(&self) -> PathBuf {
        if let Some(gobin) = &self.gobin {
            return gobin.clone();
        }
        let out = match Exec::cmd("go")
            .args(&["env", "GOBIN"])
            .stdout(Redirection::Pipe)
            .capture()
        {
            Ok(o) => o.stdout_str(),
            Err(_) => String::from(""),
        };
        if !out.trim().is_empty() {
            return PathBuf::from(out);
        }
        self.gopath_dir().join("bin")
    }

    fn gopath_dir(&self) -> PathBuf {
        if let Some(gopath) = &self.gopath {
            return gopath.clone();
        }
        let out = match Exec::cmd("go")
            .args(&["env", "GOPATH"])
            .stdout(Redirection::Pipe)
            .capture()
        {
            Ok(o) => o.stdout_str(),
            Err(_) => String::from(""),
        };
        if !out.trim().is_empty() {
            return PathBuf::from(out);
        }
        dirs::home_dir().expect("unable to determine home directory")
    }
}

pub type Result = std::result::Result<Status, Error>;

const GO_EXT: &str = ".go";

/// attempt to guess the executable binary's name from the package/target
fn executable_name<S>(target: S) -> String
where
    S: AsRef<str>,
{
    let p = PathBuf::from(target.as_ref().trim_end_matches('/').trim_end_matches('.'));
    let name = p
        .file_name()
        // expect: shouldn't panic because of the previous trims
        .expect("no file name, even though we trimmed slashes and dots")
        .to_string_lossy()
        .into_owned();

    // checking to see if there's a version segment we need to trim
    let mut chars = name.chars();
    if chars.next() == Some('v') {
        if let Some(second) = chars.next() {
            if second.is_digit(10)
                && (Version::parse(&name[1..]).is_ok() || VersionReq::parse(&name[1..]).is_ok())
            {
                // TODO: properly pre-validate / handle errors without expect/panic
                let p = p
                    .parent()
                    .expect("package needs to be more than just the version-suffix");
                return executable_name(p.to_string_lossy().into_owned());
            }
        }
    }

    // not using PathBuf::file_stem here, because that would impact more than ".go"
    if name.ends_with(GO_EXT) {
        return String::from(&name[..name.len() - GO_EXT.len()]);
    }

    name
}

#[cfg(test)]
mod tests {
    use std::fs::metadata;

    use super::super::file::tests::temp_dir;

    use super::*;

    const SAMPLE_BIN: &str = "hello-world-go";
    const SAMPLE_PKG: &str = "gitlab.com/jokeyrhyme/hello-world-go";

    #[test]
    fn gobin_values() {
        let installer = InstallerGo {
            gobin: Some(PathBuf::from("gobin/dir")),
            ..Default::default()
        };
        assert_eq!(installer.gobin_dir(), PathBuf::from("gobin/dir"));

        // TODO: consider testing fallback to `go env`
    }

    #[test]
    fn gopath_values() {
        let installer = InstallerGo {
            gopath: Some(PathBuf::from("gopath/dir")),
            ..Default::default()
        };
        assert_eq!(installer.gopath_dir(), PathBuf::from("gopath/dir"));

        // TODO: consider testing fallback to `go env`
    }

    #[test]
    fn executable_name_cases() {
        let cases = vec![
            (SAMPLE_PKG, SAMPLE_BIN),                                 // directory
            ("github.com/praetorian-inc/gokart/cmf/scan.go", "scan"), // file
            ("github.com/zricethezav/gitleaks/v7", "gitleaks"),       // versioned
            ("mvdan.cc/gofumpt/...", "gofumpt"),                      // /...
        ];
        for (input, want) in cases {
            let got = executable_name(input);
            assert_eq!(want, got);
        }
    }

    #[test]
    fn latest_then_absent() -> std::result::Result<(), Error> {
        let tmp = temp_dir()?;
        assert!(metadata(tmp.join("bin").join(SAMPLE_BIN)).is_err());

        let latest = InstallerGo {
            gobin: Some(tmp.join("bin")),
            gopath: Some(tmp.to_path_buf()),
            packages: vec![String::from(SAMPLE_PKG)],
            state: DesiredState::Latest,
        };
        latest.execute()?;
        assert!(metadata(tmp.join("bin").join(SAMPLE_BIN)).is_ok());

        let absent = InstallerGo {
            gobin: Some(tmp.join("bin")),
            gopath: Some(tmp.to_path_buf()),
            packages: vec![String::from(SAMPLE_PKG)],
            state: DesiredState::Absent,
        };
        absent.execute()?;
        assert!(metadata(tmp.join("bin").join(SAMPLE_BIN)).is_err());

        Ok(())
    }
}
