use std::{collections::HashMap, fmt};

use camino::Utf8PathBuf;
use serde::{Deserialize, Serialize};
use subprocess::{Exec, Redirection};
use thiserror::Error as ThisError;

use super::{command, file, Status};

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(rename_all = "lowercase")]
pub enum DesiredState {
    Absent,
    Latest,
    Present,
}

// TODO: wait for https://github.com/rust-lang/rust/issues/35121
// TODO: drop this and use the std::never::Never type instead
#[derive(Debug, ThisError)]
pub enum Error {
    #[error(transparent)]
    CommandJob {
        #[from]
        source: command::Error,
    },
    #[error(transparent)]
    FileJob {
        #[from]
        source: file::Error,
    },
    #[allow(dead_code)]
    #[error("never")]
    Never,
}
impl PartialEq for Error {
    fn eq(&self, other: &Error) -> bool {
        format!("{:?}", self) == format!("{:?}", other)
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(default, rename_all = "lowercase", tag = "type")]
pub struct InstallerCargo {
    /// Crates to target?
    pub crates: Vec<String>,
    /// Custom path for `CARGO_HOME` / `CARGO_INSTALL_ROOT`?
    pub root: Option<Utf8PathBuf>,
    /// Action to perform?
    /// - [`Absent`](DesiredDate::Absent) to uninstall target [`crates`](InstallerCargo::crates)
    /// - [`Latest`](DesiredDate::Latest) to update all currently-installed crates
    /// - [`Present`](DesiredDate::Present) to install target [`crates`](InstallerCargo::crates)
    pub state: DesiredState,
}
impl Default for InstallerCargo {
    fn default() -> Self {
        Self {
            crates: vec![],
            root: None,
            state: DesiredState::Latest,
        }
    }
}
impl fmt::Display for InstallerCargo {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(
            f,
            "cargo install {} {:?} crates",
            &self.crates.len(),
            &self.state
        )
    }
}
impl InstallerCargo {
    pub fn execute(&self) -> Result {
        dbg!(&self);
        match self.state {
            DesiredState::Absent => {
                if self.crates.is_empty() {
                    return Ok(Status::NoChange(String::from("0 crates to uninstall")));
                }

                let found: Vec<String> = self.found_versions().keys().map(String::from).collect();
                let surplus: Vec<String> = self
                    .crates
                    .iter()
                    .filter(|c| !c.trim().is_empty() && found.contains(c))
                    .map(String::from)
                    .collect();
                if surplus.is_empty() {
                    return Ok(Status::NoChange(String::from("0 crates uninstalled")));
                }

                let before = format!(
                    "{} of {} crates to install",
                    surplus.len(),
                    self.crates.len()
                );
                let mut args = vec![String::from("uninstall")];
                args.extend(self.root_args());
                args.extend(surplus);
                let cmd = command::Command {
                    argv: Some(args),
                    command: String::from("cargo"),
                    ..Default::default()
                };
                cmd.execute()?;

                Ok(Status::Changed(
                    before,
                    format!("{} crates uninstalled", self.crates.len()),
                ))
            }
            DesiredState::Latest => {
                let found_before = self.found_versions();
                let found: Vec<String> = found_before.keys().map(String::from).collect();

                let before = format!("{} crates", found_before.len());
                let mut args = vec![String::from("install")];
                args.extend(self.root_args());
                args.extend(found);
                let cmd = command::Command {
                    argv: Some(args),
                    command: String::from("cargo"),
                    ..Default::default()
                };
                cmd.execute()?;

                let found_after = self.found_versions();
                let found_changed = found_after
                    .into_iter()
                    .filter(|(krate, version)| found_before.get(krate) == Some(version))
                    .count();

                Ok(if found_changed == 0 {
                    Status::NoChange(before)
                } else {
                    Status::Changed(
                        before,
                        format!("{} of {} crates updated", found_changed, self.crates.len()),
                    )
                })
            }
            DesiredState::Present => {
                if self.crates.is_empty() {
                    return Ok(Status::NoChange(String::from("0 crates to install")));
                }

                let found: Vec<String> = self.found_versions().keys().map(String::from).collect();
                let missing: Vec<String> = self
                    .crates
                    .iter()
                    .filter(|c| !c.trim().is_empty() && !found.contains(c))
                    .map(String::from)
                    .collect();
                if missing.is_empty() {
                    return Ok(Status::NoChange(format!(
                        "{} crates installed",
                        self.crates.len()
                    )));
                }

                let before = format!(
                    "{} of {} crates to install",
                    missing.len(),
                    self.crates.len()
                );
                let mut args = vec![String::from("install")];
                args.extend(self.root_args());
                args.extend(missing);
                let cmd = command::Command {
                    argv: Some(args),
                    command: String::from("cargo"),
                    ..Default::default()
                };
                cmd.execute()?;

                Ok(Status::Changed(
                    before,
                    format!("{} crates installed", self.crates.len()),
                ))
            }
        }
    }

    fn found_versions(&self) -> HashMap<String, String> {
        let mut args = vec![String::from("install"), String::from("--list")];
        args.extend(self.root_args());
        let out = match Exec::cmd("cargo")
            .args(&args)
            .stdout(Redirection::Pipe)
            .capture()
        {
            Ok(o) => o.stdout_str(),
            Err(_) => String::from(""),
        };
        if out.trim().is_empty() {
            return HashMap::new();
        }
        parse_installed(out)
    }

    fn root_args(&self) -> Vec<String> {
        match &self.root {
            Some(root) => vec![String::from("--root"), root.to_string()],
            None => Vec::new(),
        }
    }
}

pub type Result = std::result::Result<Status, Error>;

fn parse_installed<S>(stdout: S) -> HashMap<String, String>
where
    S: AsRef<str>,
{
    let re = regex::Regex::new(r"^(?P<name>\S+)\sv(?P<version>\S+):").unwrap();
    let s = stdout.as_ref();
    let mut krates: HashMap<String, String> = HashMap::new();

    for line in s.lines() {
        if let Some(caps) = re.captures(line) {
            let krate = caps.get(1).unwrap().as_str();
            let version = caps.get(2).unwrap().as_str();
            krates.insert(String::from(krate), String::from(version));
        }
    }
    krates
}

#[cfg(test)]
mod tests {
    use std::fs::metadata;

    use super::super::file::tests::temp_dir;

    use super::*;

    const SAMPLE_CRATE: &str = "ssh-sensible";

    #[test]
    fn present_then_absent() -> std::result::Result<(), Error> {
        let tmp = temp_dir()?;
        let tmp_root = Utf8PathBuf::from_path_buf(tmp.join(".cargo"))
            .expect("unable to convert temporary PathBuf");
        let tmp_bin = tmp_root.join("bin");
        assert!(metadata(&tmp_bin.join(SAMPLE_CRATE)).is_err());

        let present = InstallerCargo {
            root: Some(tmp_root.clone()),
            crates: vec![String::from(SAMPLE_CRATE)],
            state: DesiredState::Present,
        };

        let initial_versions = present.found_versions();
        assert_eq!(initial_versions.len(), 0);

        present.execute()?;

        assert!(metadata(&tmp_bin.join(SAMPLE_CRATE)).is_ok());

        let installed_versions = present.found_versions();
        assert_ne!(installed_versions.len(), 0);
        assert!(installed_versions.contains_key(SAMPLE_CRATE));

        let absent = InstallerCargo {
            root: Some(tmp_root),
            crates: vec![String::from(SAMPLE_CRATE)],
            state: DesiredState::Absent,
        };
        absent.execute()?;

        assert!(metadata(&tmp_bin.join(SAMPLE_CRATE)).is_err());

        let absent_versions = absent.found_versions();
        assert_eq!(absent_versions.len(), 0);

        Ok(())
    }

    #[test]
    fn test_parse_installed() {
        let input = "
racer v2.0.12:
    racer
rustfmt v0.10.0:
    cargo-fmt
    rustfmt
rustsym v0.3.2:
    rustsym
";
        let want: HashMap<String, String> = [
            (String::from("racer"), String::from("2.0.12")),
            (String::from("rustfmt"), String::from("0.10.0")),
            (String::from("rustsym"), String::from("0.3.2")),
        ]
        .iter()
        .cloned()
        .collect();
        assert_eq!(want, parse_installed(input));
    }
}
