use std::{
    env::consts::OS,
    fs,
    path::{Path, PathBuf},
};

use serde::Serialize;
use thiserror::Error as ThisError;

#[derive(Debug, ThisError)]
pub enum Error {
    #[error("unable to find cache_dir")]
    Cache,
    #[error("unable to find config_dir")]
    Config,
    #[error("unable to find home_dir")]
    Home,
}

#[derive(Serialize)]
pub struct Facts {
    pub cache_dir: PathBuf,
    pub config_dir: PathBuf,
    pub home_dir: PathBuf,
    pub is_distro_archlinux: bool,
    pub is_distro_armbian: bool,
    pub is_distro_raspbian: bool,
    pub is_os_linux: bool,
    pub is_os_macos: bool,
    pub is_os_windows: bool,
    pub main_file: PathBuf,
}
impl Facts {
    pub fn gather() -> Result {
        let is_os_linux = OS == "linux";
        let os_release =
            fs::read_to_string(Path::new(OS_RELEASE_PATH)).unwrap_or_else(|_| String::new());
        Ok(Self {
            cache_dir: dirs::cache_dir().ok_or(Error::Cache)?,
            config_dir: dirs::config_dir().ok_or(Error::Config)?,
            home_dir: dirs::home_dir().ok_or(Error::Home)?,
            is_distro_archlinux: is_os_linux && is_distro_archlinux(&os_release),
            is_distro_armbian: is_os_linux && is_distro_armbian(&os_release),
            is_distro_raspbian: is_os_linux && is_distro_raspbian(&os_release),
            is_os_linux,
            is_os_macos: OS == "macos",
            is_os_windows: OS == "windows",
            ..Default::default()
        })
    }
}
impl Default for Facts {
    fn default() -> Self {
        Self {
            cache_dir: PathBuf::new(),
            config_dir: PathBuf::new(),
            home_dir: PathBuf::new(),
            is_distro_archlinux: false,
            is_distro_armbian: false,
            is_distro_raspbian: false,
            is_os_linux: false,
            is_os_macos: false,
            is_os_windows: false,
            #[cfg(test)]
            main_file: PathBuf::from(env!("CARGO_MANIFEST_DIR")).join("main.toml"),
            #[cfg(not(test))]
            main_file: PathBuf::new(),
        }
    }
}

pub type Result = std::result::Result<Facts, Error>;

const OS_RELEASE_ARCHLINUX: &str = "ID=arch";
const OS_RELEASE_ARMBIAN: &str = r#"NAME="Armbian"#;
const OS_RELEASE_DEBIAN: &str = "ID=debian";
const OS_RELEASE_DEBIAN_LIKE: &str = "ID_LIKE=debian";
const OS_RELEASE_PATH: &str = "/etc/os-release";
const OS_RELEASE_RASPBIAN: &str = "ID=raspbian";

fn is_distro_archlinux(os_release: &str) -> bool {
    os_release.contains(OS_RELEASE_ARCHLINUX)
}

fn is_distro_armbian(os_release: &str) -> bool {
    os_release.contains(OS_RELEASE_DEBIAN) && os_release.contains(OS_RELEASE_ARMBIAN)
}

fn is_distro_raspbian(os_release: &str) -> bool {
    os_release.contains(OS_RELEASE_DEBIAN_LIKE) && os_release.contains(OS_RELEASE_RASPBIAN)
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn gather_result_ok() {
        let got = Facts::gather();
        assert!(got.is_ok());
    }
}
