use std::{
    fmt, fs, io,
    num::NonZeroU32,
    path::{Path, PathBuf},
};

use git_url::Url;
use serde::{
    de::{Deserializer, Error as SerdeDeError},
    ser::Serializer,
    Deserialize, Serialize,
};
use subprocess::{Exec, PopenError, Redirection};
use thiserror::Error as ThisError;

use super::{command, file, Status};

#[derive(Debug, ThisError)]
pub enum Error {
    #[error(transparent)]
    Command {
        #[from]
        source: PopenError,
    },
    #[error(transparent)]
    CommandJob {
        #[from]
        source: command::Error,
    },
    #[error("{} already exists", dest.display())]
    DestExists { dest: PathBuf },
    #[error("{} not found", dest.display())]
    DestNotFound { dest: PathBuf },
    #[error(transparent)]
    FileJob {
        #[from]
        source: file::Error,
    },
    #[error("working `git` not found")]
    GitNotFound,
    #[error(transparent)]
    Io {
        #[from]
        source: io::Error,
    },
    #[allow(dead_code)]
    #[error("never")]
    Never,
    #[error(transparent)]
    UrlParse {
        #[from]
        source: git_url::parse::Error,
    },
}
impl PartialEq for Error {
    fn eq(&self, other: &Error) -> bool {
        format!("{:?}", self) == format!("{:?}", other)
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(default, rename_all = "lowercase", tag = "type")]
pub struct Git {
    /// Should we clone [`repo`](Git::repo) if it does not yet exist at [`depth`](Git::depth)?
    /// Default = yes.
    pub clone: Option<bool>,
    /// Limit fetching to a specified number of commits in history per [`git fetch --depth=N`](https://git-scm.com/docs/fetch-options#Documentation/fetch-options.txt---depthltdepthgt).
    /// Default = no limit, complete history.
    pub depth: Option<NonZeroU32>,
    /// Checkout [`repo`](Git::repo) into this target path.
    pub dest: PathBuf,
    // refspec: String
    /// Should we delete any unexpected files at [`dest`](Git::dest) if necessary?
    /// Default = no, exit early with an error instead of deleting files.
    pub force: Option<bool>,
    /// Should we pull newer commits from the origin?
    /// Default = yes, keep [`dest`](Git::dest) up to date.
    pub update: Option<bool>,
    #[serde(
        deserialize_with = "from_toml_git_url",
        serialize_with = "to_toml_git_url"
    )]
    pub repo: Url,
}
impl Default for Git {
    fn default() -> Self {
        Self {
            clone: Some(true),
            depth: None,
            dest: PathBuf::new(),
            force: None,
            repo: git_url::parse("https://gitlab.com/".as_bytes())
                .expect("unable to parse default URL"),
            update: Some(true),
        }
    }
}
impl fmt::Display for Git {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{:?} -> {:?}", &self.repo, &self.dest)
    }
}
impl Git {
    pub fn execute(&self) -> Result {
        if !has_git() {
            return Err(Error::GitNotFound);
        }

        if !self.dest.exists() && !self.clone.unwrap_or(true) {
            return Err(Error::DestNotFound {
                dest: self.dest.clone(),
            });
        }

        let mut before = String::from("absent");
        if self.dest.exists() {
            if is_git_repository(&self.dest)
                && is_git_url_eq(&git_origin_remote_url(&self.dest)?, &self.repo)
            {
                if !self.update.unwrap_or(true) {
                    return Ok(Status::NoChange(current_commit(&self.dest)?));
                }
                return self.git_pull();
            }
            if self.force.unwrap_or(false) {
                before = String::from("not repository");
                if self.dest.is_dir() {
                    fs::remove_dir_all(&self.dest)?;
                } else {
                    fs::remove_file(&self.dest)?;
                }
            } else {
                return Err(Error::DestExists {
                    dest: self.dest.clone(),
                });
            }
        }

        self.git_clone()?;

        Ok(Status::Changed(before, current_commit(&self.dest)?))
    }

    fn git_clone(&self) -> std::result::Result<(), Error> {
        let mut argv: Vec<String> = vec![String::from("clone")];
        if let Some(depth) = self.depth {
            argv.push(String::from("--depth"));
            argv.push(format!("{}", depth));
        }
        argv.push(format!("{}", self.repo));
        argv.push(String::from(self.dest.to_string_lossy()));

        let clone = command::Command {
            argv: Some(argv),
            command: String::from("git"),
            ..Default::default()
        };
        clone.execute()?;
        Ok(())
    }

    fn git_fetch(&self) -> Result {
        let before = current_commit(&self.dest)?;

        let mut argv: Vec<String> = vec![String::from("fetch")];
        if let Some(depth) = self.depth {
            argv.push(String::from("--depth"));
            argv.push(format!("{}", depth));
        }

        let fetch = command::Command {
            argv: Some(argv),
            chdir: Some(self.dest.clone()),
            command: String::from("git"),
            ..Default::default()
        };
        fetch.execute()?;

        let after = current_commit(&self.dest)?;

        status(before, after)
    }

    fn git_hard_reset(&self) -> Result {
        let before = current_commit(&self.dest)?;

        let reset = command::Command {
            argv: Some(vec![
                String::from("reset"),
                String::from("--hard"),
                String::from("FETCH_HEAD"),
            ]),
            chdir: Some(self.dest.clone()),
            command: String::from("git"),
            ..Default::default()
        };
        reset.execute()?;

        let after = current_commit(&self.dest)?;

        status(before, after)
    }

    fn git_pull(&self) -> Result {
        let before = current_commit(&self.dest)?;

        let mut argv: Vec<String> = vec![String::from("pull")];
        if let Some(depth) = self.depth {
            argv.push(String::from("--depth"));
            argv.push(format!("{}", depth));
        }

        let pull = command::Command {
            argv: Some(argv),
            chdir: Some(self.dest.clone()),
            command: String::from("git"),
            ..Default::default()
        };
        if pull.execute().is_err() && self.force.unwrap_or(false) {
            self.git_fetch()?;
            self.git_hard_reset()?;
        }

        let after = current_commit(&self.dest)?;

        status(before, after)
    }
}

pub type Result = std::result::Result<Status, Error>;

fn current_commit<P>(path: P) -> std::result::Result<String, Error>
where
    P: AsRef<Path>,
{
    let p = path.as_ref();
    let capture = Exec::cmd("git")
        .args(&["rev-parse", "--short", "HEAD"])
        .cwd(&p)
        .stdout(Redirection::Pipe)
        .capture()?;
    Ok(capture.stdout_str())
}

fn from_toml_git_url<'de, D>(deserializer: D) -> std::result::Result<Url, D::Error>
where
    D: Deserializer<'de>,
{
    let s = toml::value::Value::deserialize(deserializer)?
        .try_into::<String>()
        .map_err(SerdeDeError::custom)?;
    let u = Url::from_bytes(s.as_bytes()).map_err(SerdeDeError::custom)?;
    Ok(u)
}

fn git_origin_remote_url<P>(path: P) -> std::result::Result<Url, Error>
where
    P: AsRef<Path>,
{
    let p = path.as_ref();
    let capture = Exec::cmd("git")
        .args(&["remote", "get-url", "origin"])
        .cwd(&p)
        .stdout(Redirection::Pipe)
        .capture()?;
    if capture.success() {
        let u = git_url::parse(&capture.stdout)?;
        Ok(u)
    } else {
        Err(Error::CommandJob {
            source: command::Error::NonZeroExitStatus {
                cmd: String::from("git"),
            },
        })
    }
}

fn has_git() -> bool {
    match Exec::cmd("git").args(&["--version"]).join() {
        Ok(status) => status.success(),
        Err(_) => false,
    }
}

fn is_git_repository<P>(path: P) -> bool
where
    P: AsRef<Path>,
{
    let p = path.as_ref();
    if !p.is_dir() {
        return false;
    }
    match Exec::cmd("git").args(&["status"]).cwd(&p).join() {
        Ok(status) => status.success(),
        Err(_) => false,
    }
}

fn is_git_url_eq(a: &Url, b: &Url) -> bool {
    if a == b {
        return true;
    }
    normalize_git_url(a) == normalize_git_url(b)
}

fn normalize_git_url(u: &Url) -> Url {
    let mut n = u.clone();
    if n.scheme == git_url::Scheme::Ssh {
        // "git@...:..." is common enough that the comparison is meaningful without it
        if n.user == Some(String::from("git")) {
            n.user = None;
        }

        // some SSH URLs have a path with a leading-slash,
        // which results in a double-slash after parsing
        let p = n.path.to_string();
        if p.starts_with("//") {
            n.path = p.replace("//", "/").into();
        }
    }
    if n.scheme != git_url::Scheme::Https {
        n.scheme = git_url::Scheme::Https;
    }
    n
}

fn status(before: String, after: String) -> Result {
    if before == after {
        Ok(Status::NoChange(before))
    } else {
        Ok(Status::Changed(before, after))
    }
}

fn to_toml_git_url<S>(input: &Url, serializer: S) -> std::result::Result<S::Ok, S::Error>
where
    S: Serializer,
{
    serializer.serialize_str(&format!("{}", input))
}

#[cfg(test)]
mod tests {
    use super::super::file::tests::{temp_dir, temp_file};

    use super::*;

    const DOTFILES_REPOSITORY: &str = "https://gitlab.com/jokeyrhyme/dotfiles.git";

    #[test]
    fn error_when_absent_and_no_clone() -> std::result::Result<(), Error> {
        let git = Git {
            clone: Some(false),
            dest: temp_dir()?.join("absent"),
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        match git.execute() {
            Ok(_) => unreachable!(),
            Err(e) => assert_eq!(e, Error::DestNotFound { dest: git.dest }),
        }
        Ok(())
    }

    #[test]
    fn git_pull_when_exists_and_matching_remote() -> std::result::Result<(), Error> {
        // TODO: get checkout main HEAD^1 and get commit hash
        // TODO: retrieve commit hash for main HEAD
        let git = Git {
            dest: temp_dir()?.join("exists"),
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        let got = git.execute()?;

        // TODO: assert that resulting status includes expected old HEAD and new HEAD commit hashes
        match got {
            Status::Changed(_, _) => {}
            _ => unreachable!(),
        }
        Ok(())
    }

    #[test]
    fn rm_and_git_clone_when_not_repo_and_force() -> std::result::Result<(), Error> {
        let f = file::File {
            path: temp_file()?.to_path_buf(),
            state: file::FileState::Directory,
            ..Default::default()
        };
        f.execute()?;

        let git = Git {
            dest: f.path,
            force: Some(true),
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        let got = git.execute()?;

        // TODO: retrieve commit hash for remote main HEAD
        // TODO: assert that resulting status includes expected HEAD commit hash
        match got {
            Status::Changed(before, _) => assert_eq!(before, String::from("not repository")),
            _ => unreachable!(),
        }
        Ok(())
    }

    #[test]
    fn rm_and_git_clone_when_no_matching_remote_and_force() -> std::result::Result<(), Error> {
        let mut git = Git {
            dest: temp_dir()?.join("mismatch"),
            repo: git_url::parse(DOTFILES_REPOSITORY.as_bytes())
                .expect("unable to parse dotfiles URL"),
            ..Default::default()
        };
        git.execute()?;

        git = Git {
            dest: git.dest,
            force: Some(true),
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        let got = git.execute()?;

        // TODO: retrieve commit hash for remote main HEAD
        // TODO: assert that resulting status includes expected HEAD commit hash
        match got {
            Status::Changed(before, _) => assert_eq!(before, String::from("not repository")),
            _ => unreachable!(),
        }
        Ok(())
    }

    #[test]
    fn error_when_not_repo_and_no_force() -> std::result::Result<(), Error> {
        let f = file::File {
            path: temp_file()?.to_path_buf(),
            state: file::FileState::Directory,
            ..Default::default()
        };
        f.execute()?;

        let git = Git {
            dest: f.path,
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        match git.execute() {
            Ok(_) => unreachable!(),
            Err(e) => assert_eq!(e, Error::DestExists { dest: git.dest }),
        }
        Ok(())
    }

    #[test]
    fn error_when_no_matching_remote_and_no_force() -> std::result::Result<(), Error> {
        let mut git = Git {
            dest: temp_dir()?.join("mismatch"),
            repo: git_url::parse(DOTFILES_REPOSITORY.as_bytes())
                .expect("unable to parse dotfiles URL"),
            ..Default::default()
        };
        git.execute()?;

        git = Git {
            dest: git.dest,
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        match git.execute() {
            Ok(_) => unreachable!(),
            Err(e) => assert_eq!(e, Error::DestExists { dest: git.dest }),
        }
        Ok(())
    }

    #[test]
    fn git_clone_when_absent() -> std::result::Result<(), Error> {
        let git = Git {
            dest: temp_dir()?.join("absent"),
            repo: git_url::parse(env!("CARGO_PKG_REPOSITORY").as_bytes())
                .expect("unable to parse project URL"),
            ..Default::default()
        };

        let got = git.execute()?;

        // TODO: retrieve commit hash for remote main HEAD
        // TODO: assert that resulting status includes expected HEAD commit hash
        match got {
            Status::Changed(before, _) => assert_eq!(before, String::from("absent")),
            _ => unreachable!(),
        }
        Ok(())
    }

    #[test]
    fn is_git_url_eq_with_same_urls() {
        assert!(is_git_url_eq(
            &Url::from_bytes("https://gitlab.com/jokeyrhyme/tuning.git".as_bytes())
                .expect("unable to parse test URL"),
            &Url::from_bytes("https://gitlab.com/jokeyrhyme/tuning.git".as_bytes())
                .expect("unable to parse test URL")
        ));
    }

    #[test]
    fn is_git_url_eq_with_ssh_versus_https_urls() {
        assert!(is_git_url_eq(
            &Url::from_bytes("git@gitlab.com:/jokeyrhyme/tuning.git".as_bytes())
                .expect("unable to parse test URL"),
            &Url::from_bytes("https://gitlab.com/jokeyrhyme/tuning.git".as_bytes())
                .expect("unable to parse test URL")
        ));
    }
}
