pub(crate) mod command;
pub(crate) mod fake;
pub(crate) mod file;
pub(crate) mod git;

use std::{cell::Cell, convert::TryFrom, fmt, time::Instant};

use colored::*;
use serde::{Deserialize, Serialize};
use thiserror::Error as ThisError;

use command::Command;
use fake::Fake;
use file::File;
use git::Git;

#[derive(Debug, ThisError)]
pub enum Error {
    #[error(transparent)]
    CommandJob {
        #[from]
        source: command::Error,
    },
    #[error(transparent)]
    FakeJob {
        #[from]
        source: fake::Error,
    },
    #[error(transparent)]
    FileJob {
        #[from]
        source: file::Error,
    },
    #[error(transparent)]
    GitJob {
        #[from]
        source: git::Error,
    },
    #[error(transparent)]
    ParseToml {
        #[from]
        source: toml::de::Error,
    },
    #[allow(dead_code)] // TODO: fake test-only errors should not be here
    #[error("fake test-only error")]
    SomethingBad,
}

pub trait Execute {
    fn execute(&self) -> Result;
    fn needs(&self) -> Vec<String>;
    fn when(&self) -> bool;
}

#[derive(Debug, PartialEq)]
pub struct History {
    pub created: Instant,
    pub updated: Cell<Instant>,
}
impl Default for History {
    fn default() -> Self {
        let now = Instant::now();
        Self {
            created: now,
            updated: Cell::new(now),
        }
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(default, rename_all = "lowercase", tag = "type")]
pub struct Job {
    #[serde(skip)]
    pub history: History,

    #[serde(flatten)]
    pub metadata: Metadata,

    #[serde(flatten)]
    pub spec: Spec,
}
impl Default for Job {
    fn default() -> Self {
        Self {
            history: History::default(),
            metadata: Metadata::default(),
            spec: Spec::Fake(Fake::default()),
        }
    }
}
impl Execute for Job {
    fn execute(&self) -> Result {
        let result = match &self.spec {
            Spec::Command(j) => j.execute().map_err(|e| Error::CommandJob { source: e }),
            Spec::Fake(j) => j.execute().map_err(|e| Error::FakeJob { source: e }),
            Spec::File(j) => j.execute().map_err(|e| Error::FileJob { source: e }),
            Spec::Git(j) => j.execute().map_err(|e| Error::GitJob { source: e }),
        };
        self.history.updated.set(Instant::now());
        result
    }
    fn needs(&self) -> Vec<String> {
        self.metadata.needs.clone().unwrap_or_default()
    }
    fn when(&self) -> bool {
        self.metadata.when
    }
}
impl fmt::Display for Job {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let name = self
            .metadata
            .name
            .clone()
            .unwrap_or_else(|| format!("{}", &self.spec));
        write!(f, "{}", name)
    }
}
impl Job {
    pub fn name(&self) -> String {
        format!("{}", &self)
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(default)]
pub struct Metadata {
    pub name: Option<String>,
    pub needs: Option<Vec<String>>,
    pub when: bool,
}
impl Default for Metadata {
    fn default() -> Self {
        Self {
            name: None,
            needs: None,
            when: true,
        }
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(rename_all = "lowercase", tag = "type")]
pub enum Spec {
    Command(Command),
    Fake(Fake),
    File(File),
    Git(Git),
}
impl fmt::Display for Spec {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match &self {
            Spec::Command(j) => write!(f, "{}", j),
            Spec::Fake(j) => write!(f, "{}", j),
            Spec::File(j) => write!(f, "{}", j),
            Spec::Git(j) => write!(f, "{}", j),
        }
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
pub struct Main {
    pub jobs: Vec<Job>,
}
impl TryFrom<&str> for Main {
    type Error = Error;
    fn try_from(s: &str) -> std::result::Result<Self, Self::Error> {
        toml::from_str(s).map_err(|e| Error::ParseToml { source: e })
    }
}

pub type Result = std::result::Result<Status, Error>;
pub fn result_display(result: &Result) -> String {
    match result {
        Ok(s) => format!("{}", s),
        Err(e) => format!("{:#?}", e).red().to_string(),
    }
}
/// Returns `true` if the job is successful and should no longer block dependant jobs.
pub fn is_result_settled(result: &Result) -> bool {
    match result {
        Ok(s) => s.is_settled(),
        Err(_) => true,
    }
}
/// Returns `true` if the job has reached a terminal state and will never change state again.
pub fn is_result_done(result: &Result) -> bool {
    match result {
        Ok(s) => s.is_done(),
        Err(_) => false,
    }
}

#[derive(Clone, Debug, Deserialize, PartialEq, Serialize)]
#[serde(rename_all = "lowercase")]
pub enum Status {
    /// Job has [`needs`](Metadata::needs) that can never reach [`Done`](Status::Done).
    /// Terminal state.
    Blocked,
    /// Job is [`Done`](Status::Done) but did not result in any changes.
    /// Terminal state.
    Changed(String, String),
    /// Job is complete.
    /// Terminal state.
    Done,
    /// Runner is executing this job.
    /// Possible next states: [`Changed`](Status::Changed), [`Done`](Status::Done), [`Error`](Error), [`NoChange`](Status::NoChange).
    InProgress,
    /// Job is [`Done`](Status::Done) after making necessary changes.
    /// Terminal state.
    NoChange(String),
    /// Job has no [`needs`](Metadata::needs) or all of them are [`Done`](Status::Done).
    /// Possible next states: [`InProgress`](Status::InProgress).
    Pending,
    /// Job has a [`when`](Metadata::when) that evaluated to `false`.
    /// Terminal state.
    Skipped,
    /// Job has [`needs`](Metadata::needs) that are not yet [`Done`](Status::Done).
    /// Possible next states: [`Blocked`](Status::Blocked), [`Pending`](Status::Pending).
    Waiting,
}
impl fmt::Display for Status {
    // TODO: should Display include terminal output concerns?
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::Blocked => write!(f, "{}", "blocked".red().dimmed()),
            Self::Changed(from, to) => write!(
                f,
                "{}: {} => {}",
                "changed".yellow(),
                from.yellow().dimmed(),
                to.yellow()
            ),
            Self::Done => write!(f, "{}", "done".blue()),
            Self::InProgress => write!(f, "{}", "inprogress".cyan()),
            Self::NoChange(s) => write!(f, "{}: {}", "nochange".green(), s.green()),
            Self::Pending => write!(f, "{}", "pending".white()),
            Self::Skipped => write!(f, "{}", "skipped".blue()),
            Self::Waiting => write!(f, "{}", "waiting".white()),
        }
    }
}
impl Status {
    /// Returns `true` if the job is successful and should no longer block dependant jobs.
    pub fn is_done(&self) -> bool {
        match &self {
            Self::Changed(_, _) | Self::Done | Self::NoChange(_) => true,
            Self::Blocked | Self::InProgress | Self::Pending | Self::Skipped | Self::Waiting => {
                false
            }
        }
    }

    /// Returns `true` if the job has reached a terminal state and will never change state again.
    pub fn is_settled(&self) -> bool {
        match &self {
            Self::Blocked
            | Self::Changed(_, _)
            | Self::Done
            | Self::NoChange(_)
            | Self::Skipped => true,
            Self::InProgress | Self::Pending | Self::Waiting => false,
        }
    }
}

#[cfg(test)]
mod tests {
    use std::{path::PathBuf, time::Duration};

    use git_url::Url;

    use file::FileState;

    use super::*;

    #[test]
    fn command_toml() -> std::result::Result<(), Error> {
        let input = r#"
            [[jobs]]
            name = "run something"
            type = "command"
            command = "something"
            argv = [ "foo" ]
            "#;

        let got = Main::try_from(input)?;

        let want = Main {
            jobs: vec![Job {
                history: History::default(),
                metadata: Metadata {
                    name: Some(String::from("run something")),
                    ..Default::default()
                },
                spec: Spec::Command(Command {
                    argv: Some(vec![String::from("foo")]),
                    command: String::from("something"),
                    ..Default::default()
                }),
            }],
        };

        assert_eq!(got.jobs.len(), 1);
        // TODO: find an approach that enables reliable `history` comparison
        assert_eq!(got.jobs[0].metadata, want.jobs[0].metadata);
        assert_eq!(got.jobs[0].spec, want.jobs[0].spec);

        Ok(())
    }

    #[test]
    fn fake_toml() -> std::result::Result<(), Error> {
        let input = r#"
            [[jobs]]
            name = "fake"
            type = "fake"
            sleep_ms = 5000
            "#;

        let got = Main::try_from(input)?;

        let want = Main {
            jobs: vec![Job {
                history: History {
                    ..Default::default()
                },
                metadata: Metadata {
                    name: Some(String::from("fake")),
                    ..Default::default()
                },
                spec: Spec::Fake(Fake {
                    sleep_ms: Some(Duration::from_millis(5000)),
                    ..Default::default()
                }),
            }],
        };

        assert_eq!(got.jobs.len(), 1);
        // TODO: find an approach that enables reliable `history` comparison
        assert_eq!(got.jobs[0].metadata, want.jobs[0].metadata);
        assert_eq!(got.jobs[0].spec, want.jobs[0].spec);

        Ok(())
    }

    #[test]
    fn file_toml() -> std::result::Result<(), Error> {
        let input = r#"
            [[jobs]]
            name = "mkdir /tmp"
            type = "file"
            path = "/tmp"
            state = "directory"
            "#;

        let got = Main::try_from(input)?;

        let want = Main {
            jobs: vec![Job {
                history: History {
                    ..Default::default()
                },
                metadata: Metadata {
                    name: Some(String::from("mkdir /tmp")),
                    ..Default::default()
                },
                spec: Spec::File(File {
                    path: PathBuf::from("/tmp"),
                    state: FileState::Directory,
                    ..Default::default()
                }),
            }],
        };

        assert_eq!(got.jobs.len(), 1);
        // TODO: find an approach that enables reliable `history` comparison
        assert_eq!(got.jobs[0].metadata, want.jobs[0].metadata);
        assert_eq!(got.jobs[0].spec, want.jobs[0].spec);

        Ok(())
    }

    #[test]
    fn git_toml() -> std::result::Result<(), Error> {
        let input = r#"
            [[jobs]]
            name = "clone tuning source repo"
            type = "git"
            dest = "/usr/src/tuning"
            repo = "https://gitlab.com/jokeyrhyme/tuning.git"
            "#;

        let got = Main::try_from(input)?;

        let want = Main {
            jobs: vec![Job {
                history: History {
                    ..Default::default()
                },
                metadata: Metadata {
                    name: Some(String::from("clone tuning source repo")),
                    ..Default::default()
                },
                spec: Spec::Git(Git {
                    dest: PathBuf::from("/usr/src/tuning"),
                    repo: Url::from_bytes("https://gitlab.com/jokeyrhyme/tuning.git".as_bytes())
                        .expect("unable to parse test URL"),
                    ..Default::default()
                }),
            }],
        };

        assert_eq!(got.jobs.len(), 1);
        // TODO: find an approach that enables reliable `history` comparison
        assert_eq!(got.jobs[0].metadata, want.jobs[0].metadata);
        assert_eq!(got.jobs[0].spec, want.jobs[0].spec);

        Ok(())
    }

    #[test]
    fn absent_when_defaults_to_true() -> std::result::Result<(), Error> {
        let input = r#"
            [[jobs]]
            name = "run something"
            type = "command"
            command = "something"
            "#;

        let got = Main::try_from(input)?;

        let want = Main {
            jobs: vec![Job {
                history: History {
                    ..Default::default()
                },
                metadata: Metadata {
                    name: Some(String::from("run something")),
                    when: true,
                    ..Default::default()
                },
                spec: Spec::Command(Command {
                    command: String::from("something"),
                    ..Default::default()
                }),
            }],
        };

        assert_eq!(got.jobs.len(), 1);
        // TODO: find an approach that enables reliable `history` comparison
        assert_eq!(got.jobs[0].metadata, want.jobs[0].metadata);
        assert_eq!(got.jobs[0].spec, want.jobs[0].spec);

        Ok(())
    }
}
