use std::{
    collections::{BTreeMap, HashMap},
    sync::{Arc, Mutex},
    thread,
    time::{Duration, Instant},
};

use thiserror::Error as ThisError;

use crate::jobs::{self, is_result_done, is_result_settled, Execute, Job, Status};

// TODO: detect number of CPUs
const MAX_THREADS: usize = 2;

const REPORTER_DEBOUNCE_SEC: u64 = 2;
const REPORTER_SLEEP_MS: u64 = 200;

#[derive(Debug, ThisError)]
pub enum Error {
    #[error(transparent)]
    Job {
        #[from]
        source: jobs::Error,
    },
}

// pub type Result = std::result::Result<(), Error>;

// TODO: consider extracting the concern of println!ing Status
pub fn run(jobs: Vec<Job>) -> Vec<Job> {
    let mut results = HashMap::<String, jobs::Result>::new();
    results.reserve(jobs.len());
    // ensure every job has a registered Status
    for job in &jobs {
        results.insert(job.name(), Ok(Status::Waiting));
    }

    let jobs = Arc::new(Mutex::new(jobs));
    let results = Arc::new(Mutex::new(results));
    let mut handles = Vec::<thread::JoinHandle<_>>::with_capacity(MAX_THREADS);

    for t in 0..MAX_THREADS {
        let jobs = jobs.clone();
        let results = results.clone();

        let handle = thread::Builder::new()
            .name(format!("{}", t))
            .spawn(move || {
                loop {
                    let current_job;
                    {
                        // acquire locks
                        let mut jobs = jobs.lock().unwrap();
                        let mut results = results.lock().unwrap();

                        // check if any job statuses can be updated
                        for job in jobs.iter() {
                            let name = job.name();

                            if !job.when() {
                                results.insert(name.clone(), Ok(Status::Skipped));
                            }

                            if is_equal_status(results.get(&name).unwrap(), &Status::Waiting) {
                                // move Waiting jobs with satisfied needs over to Pending
                                let is_now_pending = job.needs().is_empty()
                                    || job
                                        .needs()
                                        .iter()
                                        .all(|n| is_result_done(results.get(n).unwrap()));
                                if is_now_pending {
                                    results.insert(name.clone(), Ok(Status::Pending));
                                }

                                // move Waiting jobs with never-met needs over to Blocked
                                let is_now_blocked = job.needs().iter().any(|n| {
                                    let needed_result = results.get(n).unwrap_or_else(|| {
                                        panic!(
                                            "cannot find result for '{}' in {:?} {:?}",
                                            &n, &results, &jobs
                                        )
                                    });
                                    is_result_settled(&needed_result)
                                        && !is_result_done(&needed_result)
                                });
                                if is_now_blocked {
                                    results.insert(name.clone(), Ok(Status::Blocked));
                                }
                            }
                        }

                        // check exit/terminate condition for thread
                        if is_all_settled(&results) {
                            return; // nothing left to do
                        }
                        // there must be at least one available job

                        // cherry-pick first available job
                        let index = match jobs.iter().position(|job| {
                            let name = job.name();
                            // this .unwrap() is fine, as all jobs have a registered Status
                            is_equal_status(results.get(&name).unwrap(), &Status::Pending)
                        }) {
                            Some(i) => i,
                            None => {
                                // the only remaining jobs must already be InProgress
                                // nothing left to do
                                return;
                            }
                        };
                        current_job = jobs.remove(index);
                        let name = current_job.name();
                        results.insert(name.clone(), Ok(Status::InProgress));

                        // release/drop locks
                    }

                    // execute job
                    let name = current_job.name();
                    let result = current_job.execute();

                    // record result of job
                    {
                        // acquire locks
                        let mut jobs = jobs.lock().unwrap();
                        let mut results = results.lock().unwrap();

                        jobs.push(current_job);
                        results.insert(name.clone(), result);
                        println!(
                            "job: {}: {}",
                            &name,
                            jobs::result_display(results.get(&name).unwrap())
                        );
                        // release/drop locks
                    }
                }
            })
            .unwrap_or_else(|err| panic!("unable to spawn thread {}: {:?}", t, err));
        handles.push(handle);
    }

    let reporter_handle;
    {
        let results = results.clone();
        reporter_handle = thread::Builder::new()
            .name(String::from("reporter"))
            .spawn(move || {
                let mut last_reported = Instant::now();
                loop {
                    {
                        // acquire locks
                        let results = results.lock().unwrap();

                        if last_reported.elapsed() > Duration::from_secs(REPORTER_DEBOUNCE_SEC) {
                            report_results(&results);
                            last_reported = Instant::now();
                        }

                        // check exit/terminate condition for thread
                        if is_all_settled(&results) {
                            return; // nothing left to do
                        }

                        // release/drop locks
                    }

                    thread::sleep(Duration::from_millis(REPORTER_SLEEP_MS));
                }
            })
            .unwrap_or_else(|err| panic!("unable to spawn reporter thread: {:?}", err));
    }
    for handle in handles {
        handle.join().expect("worker thread failed");
    }
    reporter_handle.join().expect("reporter thread failed");
    report_results(
        &Arc::try_unwrap(results)
            .expect("leaked reference to results")
            .into_inner()
            .expect("other mutex locker panicked"),
    );

    Arc::try_unwrap(jobs)
        .expect("leaked reference to jobs")
        .into_inner()
        .expect("other mutex locker panicked")
}

fn is_all_settled(results: &HashMap<String, jobs::Result>) -> bool {
    results.iter().all(|(_, result)| is_result_settled(result))
}

fn is_equal_status(result: &jobs::Result, status: &Status) -> bool {
    match result {
        Ok(s) => s == status,
        Err(_) => false,
    }
}

fn report_results(results: &HashMap<String, jobs::Result>) {
    let mut total: u64 = 0;
    let mut summary = BTreeMap::<String, u64>::new();
    for result in results.values() {
        let key = String::from(match result {
            Ok(Status::Blocked) => "blocked",
            Ok(Status::Changed(_, _)) => "changed",
            Ok(Status::Done) => "done",
            Ok(Status::InProgress) => "inprogress",
            Ok(Status::NoChange(_)) => "nochange",
            Ok(Status::Pending) => "pending",
            Ok(Status::Skipped) => "skipped",
            Ok(Status::Waiting) => "waiting",
            Err(_) => "error",
        });

        let count = summary.entry(key).or_insert(0);
        *count += 1;
        total += 1;
    }

    println!("jobs={} summary={:?}", total, summary);
}

#[cfg(test)]
mod tests {
    use std::time::Duration;

    use super::*;

    use crate::jobs::{fake::Fake, Metadata, Spec};

    fn assert_executed(job: &Job) -> bool {
        job.history.created != job.history.updated.get()
    }

    fn assert_not_executed(job: &Job) -> bool {
        job.history.created == job.history.updated.get()
    }

    fn find_job<S>(jobs: &mut Vec<Job>, name: S) -> Job
    where
        S: AsRef<str>,
    {
        jobs.remove(job_position(&jobs, name.as_ref()))
    }

    fn job_position<S>(jobs: &[Job], name: S) -> usize
    where
        S: AsRef<str>,
    {
        jobs.iter()
            .position(|job| job.name() == name.as_ref())
            .unwrap_or_else(|| panic!("job '{}' not found after runner", name.as_ref()))
    }

    fn make_fake_job<S>(name: S, status: Status) -> Job
    where
        S: AsRef<str>,
    {
        Job {
            metadata: Metadata {
                name: Some(String::from(name.as_ref())),
                ..Default::default()
            },
            spec: Spec::Fake(Fake {
                status: Some(status),
                ..Default::default()
            }),
            ..Default::default()
        }
    }

    fn make_fake_job_sleep<S>(name: S, sleep: Duration, status: Status) -> Job
    where
        S: AsRef<str>,
    {
        Job {
            metadata: Metadata {
                name: Some(String::from(name.as_ref())),
                ..Default::default()
            },
            spec: Spec::Fake(Fake {
                sleep_ms: Some(sleep),
                status: Some(status),
            }),
            ..Default::default()
        }
    }

    #[test]
    fn run_does_not_execute_job_with_false_when_or_needs_job_with_false_when() {
        let mut a = make_fake_job("a", Status::Done);
        let mut b = make_fake_job("b", Status::Done);
        a.metadata.when = false;
        b.metadata.needs = Some(vec![String::from("a")]);

        let mut jobs = vec![a, b];
        jobs = run(jobs);

        a = find_job(&mut jobs, "a");
        b = find_job(&mut jobs, "b");
        assert_not_executed(&a);
        assert_not_executed(&b);
    }

    #[test]
    fn run_executes_unordered_jobs() {
        const MAX_COUNT: usize = 10;
        let mut jobs = Vec::<Job>::with_capacity(MAX_COUNT);
        for i in 0..MAX_COUNT {
            let job = make_fake_job(
                format!("{}", i),
                match i % 2 {
                    0 => Status::Done,
                    _ => Status::NoChange(format!("{}", i)),
                },
            );
            jobs.push(job);
        }

        jobs = run(jobs);

        for job in jobs {
            assert_executed(&job);
        }
    }

    #[test]
    fn run_executes_unordered_jobs_concurrently() {
        let fake_sleep_ms = REPORTER_SLEEP_MS * 3;
        let mut a = make_fake_job_sleep("a", Duration::from_millis(fake_sleep_ms), Status::Done);
        let mut b = make_fake_job_sleep("b", Duration::from_millis(fake_sleep_ms), Status::Done);

        let mut jobs = vec![a, b];
        jobs = run(jobs);

        a = find_job(&mut jobs, "a");
        b = find_job(&mut jobs, "b");
        assert_not_executed(&a);
        assert_not_executed(&b);

        // assert that both jobs finished very recently,
        // that they had to have been executed concurrently
        assert!(a.history.created.elapsed() > Duration::from_millis(fake_sleep_ms));
        assert!(b.history.created.elapsed() > Duration::from_millis(fake_sleep_ms));
        assert!(a.history.updated.get().elapsed() < Duration::from_millis(REPORTER_SLEEP_MS + 50));
        assert!(b.history.updated.get().elapsed() < Duration::from_millis(REPORTER_SLEEP_MS + 50));
    }

    #[test]
    fn run_executes_jobs_with_complex_needs() {
        const MAX_COUNT: usize = 100;
        let mut jobs = Vec::<Job>::with_capacity(MAX_COUNT);
        for i in 0..MAX_COUNT {
            let mut job = make_fake_job(format!("{}", i), Status::Done);
            match i % 10 {
                2 => {
                    job.metadata.needs = Some(vec![format!("{}", i + 2)]);
                }
                3 => {
                    job.metadata.needs = Some(vec![format!("{}", i - 3)]);
                }
                4 => {
                    job.metadata.needs = Some(vec![format!("{}", i + 3)]);
                }
                7 => {
                    job.metadata.needs = Some(vec![String::from("99")]);
                }
                _ => { /* noop */ }
            }
            jobs.push(job);
        }

        jobs = run(jobs);

        for job in &jobs {
            assert_executed(&job);
            let i = job.name().parse::<usize>().unwrap_or_else(|err| {
                panic!("could not parse '{}' as usize: {:?}", job.name(), err)
            });
            match i % 10 {
                2 => {
                    let needed_job_index = job_position(&jobs, format!("{}", i + 2));
                    let needed_job = jobs
                        .get(needed_job_index)
                        .expect("job not found after runner");
                    // jobs ending in 2 should all run after the next job ending in 4
                    assert!(job.history.updated.get() > needed_job.history.updated.get());
                }
                3 => {
                    let needed_job_index = job_position(&jobs, format!("{}", i - 3));
                    let needed_job = jobs
                        .get(needed_job_index)
                        .expect("job not found after runner");
                    // jobs ending in 3 should all run after the previous job ending in 0
                    assert!(job.history.updated.get() > needed_job.history.updated.get());
                }
                4 => {
                    let needed_job_index = job_position(&jobs, format!("{}", i + 3));
                    let needed_job = jobs
                        .get(needed_job_index)
                        .expect("job not found after runner");
                    // jobs ending in 4 should all run after the next job ending in 7
                    assert!(job.history.updated.get() > needed_job.history.updated.get());
                }
                7 => {
                    let needed_job_index = job_position(&jobs, format!("{}", 99));
                    let needed_job = jobs
                        .get(needed_job_index)
                        .expect("job not found after runner");
                    // jobs ending in 7 should all run after job #99
                    assert!(job.history.updated.get() > needed_job.history.updated.get());
                }
                _ => { /* noop */ }
            }
        }
    }

    #[test]
    fn run_executes_ordered_jobs() {
        let mut a = make_fake_job("a", Status::Done);
        let mut b = make_fake_job("b", Status::Done);
        a.metadata.needs = Some(vec![String::from("b")]);

        let mut jobs = vec![a, b];
        jobs = run(jobs);

        a = find_job(&mut jobs, "a");
        b = find_job(&mut jobs, "b");
        assert_executed(&a);
        assert_executed(&b);
        // assert that "a" finished after "b"
        assert!(a.history.updated > b.history.updated);
    }

    #[test]
    fn run_does_not_execute_ordered_job_when_needs_are_not_done() {
        let mut a = make_fake_job("a", Status::Done);
        let mut b = make_fake_job("b", Status::Blocked);
        a.metadata.needs = Some(vec![String::from("b")]);

        let mut jobs = vec![a, b];
        jobs = run(jobs);

        a = find_job(&mut jobs, "a");
        b = find_job(&mut jobs, "b");
        assert_not_executed(&a);
        assert_executed(&b);
    }

    #[test]
    fn run_does_not_execute_ordered_job_when_some_needs_are_not_done() {
        let mut a = make_fake_job("a", Status::Done);
        let mut b = make_fake_job("b", Status::Blocked);
        let mut c = make_fake_job("c", Status::Done);
        a.metadata.needs = Some(vec![String::from("b"), String::from("c")]);
        b.metadata.needs = Some(vec![String::from("c")]);

        let mut jobs = vec![a, b, c];
        jobs = run(jobs);

        a = find_job(&mut jobs, "a");
        b = find_job(&mut jobs, "b");
        c = find_job(&mut jobs, "c");
        assert_not_executed(&a);
        assert_executed(&b);
        assert_executed(&c);
    }
}
