use std::{fmt, thread, time::Duration};

use serde::{
    de::{Deserializer, Error as SerdeDeError},
    Deserialize, Serialize,
};
use thiserror::Error as ThisError;

use super::Status;

// TODO: wait for https://github.com/rust-lang/rust/issues/35121
// TODO: drop this and use the std::never::Never type instead
#[derive(Debug, ThisError)]
pub enum Error {
    #[allow(dead_code)]
    #[error("never")]
    Never,
}
impl PartialEq for Error {
    fn eq(&self, other: &Error) -> bool {
        format!("{:?}", self) == format!("{:?}", other)
    }
}

#[derive(Debug, Deserialize, PartialEq, Serialize)]
#[serde(rename_all = "lowercase", tag = "type")]
pub struct Fake {
    #[serde(default, deserialize_with = "from_toml_duration_ms")]
    pub sleep_ms: Option<Duration>,
    pub status: Option<Status>,
}
impl Default for Fake {
    fn default() -> Self {
        Self {
            status: Some(Status::Done),
            sleep_ms: Some(Duration::from_millis(0)),
        }
    }
}
impl fmt::Display for Fake {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "{:?} {:?}", &self.sleep_ms, &self.status)
    }
}
impl Fake {
    pub fn execute(&self) -> Result {
        thread::sleep(self.sleep_ms.unwrap_or_else(|| Duration::from_millis(0)));
        Ok(self.status.clone().unwrap_or(Status::Done))
    }
}

pub type Result = std::result::Result<Status, Error>;

fn from_toml_duration_ms<'de, D>(deserializer: D) -> std::result::Result<Option<Duration>, D::Error>
where
    D: Deserializer<'de>,
{
    Ok(Some(Duration::from_millis(
        toml::value::Value::deserialize(deserializer)?
            .try_into::<u64>()
            .map_err(SerdeDeError::custom)?,
    )))
}
