//! Code to be shared with other CLIs. At the moment, this module is not intended to become a stable API.

pub mod midi;

use std::{
    fs::File,
    path::{Path, PathBuf},
};

use clap::Parser;
use tune::{
    key::PianoKey,
    pitch::{Ratio, RatioExpression, RatioExpressionVariant},
    scala::{self, Kbm, KbmImportError, KbmRoot, Scl, SclBuildError, SclImportError},
};

use crate::{CliError, CliResult};

#[derive(Parser)]
pub enum SclCommand {
    /// Scale with custom step sizes
    #[clap(name = "steps")]
    Steps {
        /// Steps of the scale
        #[clap(use_delimiter = true)]
        items: Vec<RatioExpression>,
    },

    /// Rank-2 temperament
    #[clap(name = "rank2")]
    Rank2Temperament {
        /// First generator (finite), e.g. 3/2
        generator: Ratio,

        /// Number of positive generations using the first generator, e.g. 6
        num_pos_generations: u16,

        /// Number of negative generations using the first generator, e.g. 1
        #[clap(default_value = "0")]
        num_neg_generations: u16,

        /// Second generator (infinite)
        #[clap(long = "per", default_value = "2")]
        period: Ratio,
    },

    /// Harmonic series
    #[clap(name = "harm")]
    HarmonicSeries {
        /// The lowest harmonic, e.g. 8
        lowest_harmonic: u16,

        /// Number of of notes, e.g. 8
        #[clap(short = 'n')]
        number_of_notes: Option<u16>,

        /// Build subharmonic series
        #[clap(long = "sub")]
        subharmonics: bool,
    },

    /// Import scl file
    #[clap(name = "import")]
    Import {
        /// The location of the file to import
        scl_file_location: PathBuf,
    },
}

impl SclCommand {
    pub fn to_scl(&self, description: Option<String>) -> Result<Scl, CliError> {
        Ok(match self {
            SclCommand::Steps { items } => create_custom_scale(description, items)?,
            &SclCommand::Rank2Temperament {
                generator,
                num_pos_generations,
                num_neg_generations,
                period,
            } => scala::create_rank2_temperament_scale(
                description,
                generator,
                num_pos_generations,
                num_neg_generations,
                period,
            )?,
            &SclCommand::HarmonicSeries {
                lowest_harmonic,
                number_of_notes,
                subharmonics,
            } => scala::create_harmonics_scale(
                description,
                lowest_harmonic,
                number_of_notes.unwrap_or(lowest_harmonic),
                subharmonics,
            )?,
            SclCommand::Import { scl_file_location } => {
                let mut scale = import_scl_file(scl_file_location)?;
                if let Some(description) = description {
                    scale.set_description(description)
                }
                scale
            }
        })
    }
}

fn create_custom_scale(
    description: impl Into<Option<String>>,
    items: &[RatioExpression],
) -> Result<Scl, SclBuildError> {
    let mut builder = Scl::builder();
    for item in items {
        match item.variant() {
            RatioExpressionVariant::Float { float_value } => {
                if let Some(float_value) = as_int(float_value) {
                    builder = builder.push_int(float_value);
                    continue;
                }
            }
            RatioExpressionVariant::Fraction { numer, denom } => {
                if let (Some(numer), Some(denom)) = (as_int(numer), as_int(denom)) {
                    builder = builder.push_fraction(numer, denom);
                    continue;
                }
            }
            _ => {}
        }
        builder = builder.push_ratio(item.ratio());
    }

    match description.into() {
        Some(description) => builder.build_with_description(description),
        None => builder.build(),
    }
}

fn as_int(float: f64) -> Option<u32> {
    let rounded = float.round();
    if (float - rounded).abs() < 1e-6 {
        Some(rounded as u32)
    } else {
        None
    }
}

#[derive(Parser)]
pub struct KbmRootOptions {
    /// Reference note that should sound at its original or a custom pitch, e.g. 69@440Hz
    ref_note: KbmRoot,

    /// root note / "middle note" of the scale if different from reference note
    #[clap(long = "root")]
    root_note: Option<i16>,
}

impl KbmRootOptions {
    pub fn to_kbm_root(&self) -> KbmRoot {
        match self.root_note {
            Some(root_note) => self
                .ref_note
                .shift_origin_by(i32::from(root_note) - self.ref_note.origin.midi_number()),
            None => self.ref_note,
        }
    }
}

#[derive(Parser)]
pub struct KbmOptions {
    #[clap(flatten)]
    kbm_root: KbmRootOptions,

    /// Lower key bound (inclusive)
    #[clap(long = "lo-key", default_value = "21")]
    lower_key_bound: i32,

    /// Upper key bound (exclusive)
    #[clap(long = "up-key", default_value = "109")]
    upper_key_bound: i32,

    /// Keyboard mapping entries, e.g. 0,x,1,x,2,3,x,4,x,5,x,6
    #[clap(long = "key-map", use_delimiter = true, parse(try_from_str=parse_item))]
    items: Option<Vec<Item>>,

    /// The formal octave of the keyboard mapping, e.g. n in n-EDO
    #[clap(long = "octave")]
    formal_octave: Option<i16>,
}

enum Item {
    Mapped(i16),
    Unmapped,
}

fn parse_item(s: &str) -> Result<Item, &'static str> {
    if ["x", "X"].contains(&s) {
        return Ok(Item::Unmapped);
    }
    if let Ok(parsed) = s.parse() {
        return Ok(Item::Mapped(parsed));
    }
    Err("Invalid keyboard mapping entry. Should be x, X or an 16-bit signed integer")
}

impl KbmOptions {
    pub fn to_kbm(&self) -> CliResult<Kbm> {
        let mut builder = Kbm::builder(self.kbm_root.to_kbm_root()).range(
            PianoKey::from_midi_number(self.lower_key_bound)
                ..PianoKey::from_midi_number(self.upper_key_bound),
        );
        if let Some(items) = &self.items {
            for item in items {
                match item {
                    &Item::Mapped(scale_degree) => {
                        builder = builder.push_mapped_key(scale_degree);
                    }
                    Item::Unmapped => {
                        builder = builder.push_unmapped_key();
                    }
                }
            }
        }
        if let Some(formal_octave) = self.formal_octave {
            builder = builder.formal_octave(formal_octave);
        }
        Ok(builder.build()?)
    }
}

pub fn import_scl_file(file_name: &Path) -> Result<Scl, String> {
    File::open(file_name)
        .map_err(SclImportError::IoError)
        .and_then(Scl::import)
        .map_err(|err| match err {
            SclImportError::IoError(err) => format!("Could not read scl file: {}", err),
            SclImportError::ParseError { line_number, kind } => format!(
                "Could not parse scl file at line {} ({:?})",
                line_number, kind
            ),
            SclImportError::StructuralError(err) => format!("Malformed scl file ({:?})", err),
            SclImportError::BuildError(err) => format!("Unsupported scl file ({:?})", err),
        })
}

pub fn import_kbm_file(file_name: &Path) -> Result<Kbm, String> {
    File::open(file_name)
        .map_err(KbmImportError::IoError)
        .and_then(Kbm::import)
        .map_err(|err| match err {
            KbmImportError::IoError(err) => format!("Could not read kbm file: {}", err),
            KbmImportError::ParseError { line_number, kind } => format!(
                "Could not parse kbm file at line {} ({:?})",
                line_number, kind
            ),
            KbmImportError::StructuralError(err) => format!("Malformed kbm file ({:?})", err),
            KbmImportError::BuildError(err) => format!("Unsupported kbm file ({:?})", err),
        })
}
