//! Error types

use anchor_lang::prelude::*;
use anchor_lang::solana_program::{decode_error::DecodeError, program_error::ProgramError};

/// Errors that may be returned by the TokenLending program.
#[error_code]
pub enum LendingError {
    // 0
    /// Invalid instruction data passed in.
    #[msg("Failed to unpack instruction data")]
    InstructionUnpackError,
    /// The account cannot be initialized because it is already in use.
    #[msg("Account is already initialized")]
    AlreadyInitialized,
    /// Lamport balance below rent-exempt threshold.
    #[msg("Lamport balance below rent-exempt threshold")]
    NotRentExempt,
    /// The program address provided doesn't match the value generated by the program.
    #[msg("Market authority is invalid")]
    InvalidMarketAuthority,
    /// Expected a different market owner
    #[msg("Market owner is invalid")]
    InvalidMarketOwner,

    // 5
    /// The owner of the input isn't set to the program address generated by the program.
    #[msg("Input account owner is not the program address")]
    InvalidAccountOwner,
    /// The owner of the account input isn't set to the correct token program id.
    #[msg("Input token account is not owned by the correct token program id")]
    InvalidTokenOwner,
    /// Expected an SPL Token account
    #[msg("Input token account is not valid")]
    InvalidTokenAccount,
    /// Expected an SPL Token mint
    #[msg("Input token mint account is not valid")]
    InvalidTokenMint,
    /// Expected a different SPL Token program
    #[msg("Input token program account is not valid")]
    InvalidTokenProgram,

    // 10
    /// Invalid amount, must be greater than zero
    #[msg("Input amount is invalid")]
    InvalidAmount,
    /// Invalid config value
    #[msg("Input config value is invalid")]
    InvalidConfig,
    /// Invalid config value
    #[msg("Input account must be a signer")]
    InvalidSigner,
    /// Invalid account input
    #[msg("Invalid account input")]
    InvalidAccountInput,
    /// Math operation overflow
    #[msg("Math operation overflow")]
    MathOverflow,

    // 15
    /// Token initialize mint failed
    #[msg("Token initialize mint failed")]
    TokenInitializeMintFailed,
    /// Token initialize account failed
    #[msg("Token initialize account failed")]
    TokenInitializeAccountFailed,
    /// Token transfer failed
    #[msg("Token transfer failed")]
    TokenTransferFailed,
    /// Token mint to failed
    #[msg("Token mint to failed")]
    TokenMintToFailed,
    /// Token burn failed
    #[msg("Token burn failed")]
    TokenBurnFailed,

    // 20
    /// Insufficient liquidity available
    #[msg("Insufficient liquidity available")]
    InsufficientLiquidity,
    /// This reserve's collateral cannot be used for borrows
    #[msg("Input reserve has collateral disabled")]
    ReserveCollateralDisabled,
    /// Reserve state stale
    #[msg("Reserve state needs to be refreshed")]
    ReserveStale,
    /// Withdraw amount too small
    #[msg("Withdraw amount too small")]
    WithdrawTooSmall,
    /// Withdraw amount too large
    #[msg("Withdraw amount too large")]
    WithdrawTooLarge,

    // 25
    /// Borrow amount too small
    #[msg("Borrow amount too small to receive liquidity after fees")]
    BorrowTooSmall,
    /// Borrow amount too large
    #[msg("Borrow amount too large for deposited collateral")]
    BorrowTooLarge,
    /// Repay amount too small
    #[msg("Repay amount too small to transfer liquidity")]
    RepayTooSmall,
    /// Liquidation amount too small
    #[msg("Liquidation amount too small to receive collateral")]
    LiquidationTooSmall,
    /// Cannot liquidate healthy obligations
    #[msg("Cannot liquidate healthy obligations")]
    ObligationHealthy,

    // 30
    /// Obligation state stale
    #[msg("Obligation state needs to be refreshed")]
    ObligationStale,
    /// Obligation reserve limit exceeded
    #[msg("Obligation reserve limit exceeded")]
    ObligationReserveLimit,
    /// Expected a different obligation owner
    #[msg("Obligation owner is invalid")]
    InvalidObligationOwner,
    /// Obligation deposits are empty
    #[msg("Obligation deposits are empty")]
    ObligationDepositsEmpty,
    /// Obligation borrows are empty
    #[msg("Obligation borrows are empty")]
    ObligationBorrowsEmpty,

    // 35
    /// Obligation deposits have zero value
    #[msg("Obligation deposits have zero value")]
    ObligationDepositsZero,
    /// Obligation borrows have zero value
    #[msg("Obligation borrows have zero value")]
    ObligationBorrowsZero,
    /// Invalid obligation collateral
    #[msg("Invalid obligation collateral")]
    InvalidObligationCollateral,
    /// Invalid obligation liquidity
    #[msg("Invalid obligation liquidity")]
    InvalidObligationLiquidity,
    /// Obligation collateral is empty
    #[msg("Obligation collateral is empty")]
    ObligationCollateralEmpty,

    // 40
    /// Obligation liquidity is empty
    #[msg("Obligation liquidity is empty")]
    ObligationLiquidityEmpty,
    /// Negative interest rate
    #[msg("Interest rate is negative")]
    NegativeInterestRate,
    /// Oracle config is invalid
    #[msg("Input oracle config is invalid")]
    InvalidOracleConfig,
    /// Not enough liquidity after flash loan
    #[msg("Not enough liquidity after flash loan")]
    NotEnoughLiquidityAfterFlashLoan,
    /// Used to prevent access to flashloan and liquidation
    #[msg("405 method not allowed")]
    MethodNotAllowed,
    /// Indicates the operation requested during a pseudo deposit update instruction is invalid
    #[msg("Invalid update pseudo deposit operation")]
    InvalidUpdatePseudoDepositOperation,
    /// Indicates the account used to sign/authorize a borrow instruction is unauthorized
    #[msg("Unauthorized borrow signer")]
    UnauthorizedBorrowAuthorizer,
    /// Indicates the given pyth price account is incorrect
    #[msg("Invalid Pyth Price Account")]
    InvalidPythPriceAccount,
}

impl From<LendingError> for ProgramError {
    fn from(e: LendingError) -> Self {
        ProgramError::Custom(e as u32)
    }
}

impl<T> DecodeError<T> for LendingError {
    fn type_of() -> &'static str {
        "Lending Error"
    }
}
