use crate::cmd::{link, relink, unlink};
use clap::{
    crate_authors, crate_description, crate_name, crate_version, AppSettings, Parser, Subcommand,
};
use tuck::{TuckDir, TuckErr};
use std::path::PathBuf;

pub trait Command {
    fn exec(self, config: Config) -> Result<(), TuckErr>;
}

#[derive(Debug, Parser)]
pub struct Config {
    /// Set stow dir to DIR (default is current dir)
    #[clap(short = 'd', long, display_order = 0)]
    pub dir: TuckDir,

    /// Set target to DIR (default is parent of stow dir)
    #[clap(short = 't', long, display_order = 1)]
    pub target: PathBuf,

    /// Enable special handling for "dotfiles" (files or folders whose name begins with a period) in the package directory.
    ///
    /// If this option is enabled, Stow will add a preprocessing step for each file or folder whose name begins with "dot-",
    /// and replace the "dot-" prefix in the name by a period (.). This is useful when Stow is used to manage collections of
    /// dotfiles, to avoid having a package directory full of hidden files.
    ///
    /// For example, suppose we have a package containing two files, stow/dot-bashrc and stow/dot-emacs.d/init.el.
    /// With this option, Stow will create symlinks from .bashrc to stow/dot-bashrc and from .emacs.d/init.el to
    /// stow/dot-emacs.d/init.el. Any other files, whose name does not begin with "dot-", will be processed as usual.
    #[clap(short = 'D', long)]
    pub dotfiles: bool,
}

#[derive(Debug, Subcommand)]
pub enum SubCmd {
    /// Stow the package names that follow this option
    #[clap(display_order = 0)]
    Link(link::Link),

    /// Unstow the package names that follow this option
    #[clap(display_order = 1)]
    Unlink(unlink::Unlink),

    /// Restow (like stow -D followed by stow -S)
    #[clap(display_order = 2)]
    Relink(relink::Relink),
}

impl Command for SubCmd {
    fn exec(self, config: Config) -> Result<(), TuckErr> {
        match self {
            Self::Link(x) => x.exec(config),
            Self::Unlink(x) => x.exec(config),
            Self::Relink(x) => x.exec(config),
        }
    }
}

#[derive(Debug, Parser)]
#[clap(
    name = crate_name!(),
    version = crate_version!(),
    about = crate_description!(),
    author = crate_authors!(),
    setting = AppSettings::DisableHelpSubcommand,
    setting = AppSettings::SubcommandRequired,
)]
pub struct Cli {
    #[clap(subcommand)]
    pub cmd: SubCmd,

    #[clap(flatten)]
    pub config: Config,
}

impl Cli {
    pub fn new() -> Self {
        Self::parse()
    }
}

// stow (GNU Stow) version 2.3.1
//
// SYNOPSIS:
//
//     stow [OPTION ...] [-D|-S|-R] PACKAGE ... [-D|-S|-R] PACKAGE ...
//
// OPTIONS:
//
//     -d DIR, --dir=DIR     Set stow dir to DIR (default is current dir)
//     -t DIR, --target=DIR  Set target to DIR (default is parent of stow dir)
//
//     -S, --stow            Stow the package names that follow this option
//     -D, --delete          Unstow the package names that follow this option
//     -R, --restow          Restow (like stow -D followed by stow -S)
//
//     --ignore=REGEX        Ignore files ending in this Perl regex
//     --defer=REGEX         Don't stow files beginning with this Perl regex
//                           if the file is already stowed to another package
//     --override=REGEX      Force stowing files beginning with this Perl regex
//                           if the file is already stowed to another package
//     --adopt               (Use with care!)  Import existing files into stow package
//                           from target.  Please read docs before using.
//     -p, --compat          Use legacy algorithm for unstowing
//
//     -n, --no, --simulate  Do not actually make any filesystem changes
//     -v, --verbose[=N]     Increase verbosity (levels are from 0 to 5;
//                             -v or --verbose adds 1; --verbose=N sets level)
//     -V, --version         Show stow version number
//     -h, --help            Show this help
//
// Report bugs to: bug-stow@gnu.org
// Stow home page: <http://www.gnu.org/software/stow/>
// General help using GNU software: <http://www.gnu.org/gethelp/>
