pub mod mem;
pub mod rtne;
pub mod arch;
pub mod exec;
pub mod ir;
pub mod bin;

pub use mem::*;
pub use rtne::*;
pub use arch::*;
pub use exec::*;
pub use bin::*;

#[cfg(test)]
mod tests {
    pub use super::*;
    
    #[test]
    fn test_immut() {
        fn test_new() {
            let immut_mem = Immut::new(12);

            let _get_ok = match immut_mem.get(11) {
                Ok(ok) => assert_eq!(ok, 0x0),
                Err(err) => panic!("{}", err)
            };

            let _get_err = match immut_mem.get(12) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            assert_eq!(immut_mem.len(), 12);
            assert_eq!(immut_mem.clone_into_vec(), vec![0x0; 12]);
        }

        fn test_from() {
            let immut_mem = Immut::from(vec![1, 2, 3, 4, 5]);

            let _get_ok = match immut_mem.get(2) {
                Ok(ok) => assert_eq!(ok, 3),
                Err(err) => panic!("{}", err)
            };

            let _get_err = match immut_mem.get(5) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            assert_eq!(immut_mem.len(), 5);
            assert_eq!(immut_mem.clone_into_vec(), vec![1, 2, 3, 4, 5]);
        }

        fn test_from_mut() {
            let mut_mem = Mut::from(vec![1, 2, 3, 4, 5]);
            let immut_mem = Immut::from_mut(mut_mem);

            let _get_ok = match immut_mem.get(2) {
                Ok(ok) => assert_eq!(ok, 3),
                Err(err) => panic!("{}", err)
            };

            let _get_err = match immut_mem.get(5) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            assert_eq!(immut_mem.len(), 5);
            assert_eq!(immut_mem.clone_into_vec(), vec![1, 2, 3, 4, 5]);
        }
        
        test_new();
        test_from();
        test_from_mut()
    }

    #[test]
    fn test_mut() {
        fn test_new() {
            let mut mut_mem = Mut::new(12);
            let index11 = '😎' as Byte;

            let _set_ok = match mut_mem.set(11, index11) {
                Ok(ok) => assert_eq!(ok, ()),
                Err(err) => panic!("{}", err)
            };

            let _set_err = match mut_mem.set(12, 0x0) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            let _get_ok = match mut_mem.get(11) {
                Ok(ok) => assert_eq!(ok, index11),
                Err(err) => panic!("{}", err)
            };

            let _get_err = match mut_mem.get(12) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            assert_eq!(mut_mem.len(), 12);
            assert_eq!(mut_mem.clone_into_vec(), vec![0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, index11]);
        }

        fn test_from() {
            let mut mut_mem = Mut::from(vec![0x0; 12]);
            let index11 = '😎' as Byte;

            let _set_ok = match mut_mem.set(11, index11) {
                Ok(ok) => assert_eq!(ok, ()),
                Err(err) => panic!("{}", err)
            };

            let _set_err = match mut_mem.set(12, 0x0) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            let _get_ok = match mut_mem.get(11) {
                Ok(ok) => assert_eq!(ok, index11),
                Err(err) => panic!("{}", err)
            };

            let _get_err = match mut_mem.get(12) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            assert_eq!(mut_mem.len(), 12);
            assert_eq!(mut_mem.clone_into_vec(), vec![0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, index11]);
        }

        fn test_from_immut() {
            let immut_mem = Immut::from(vec![1, 2, 3, 4, 5]);
            let mut_mem = Mut::from_immut(immut_mem);

            let _get_ok = match mut_mem.get(2) {
                Ok(ok) => assert_eq!(ok, 3),
                Err(err) => panic!("{}", err)
            };

            let _get_err = match mut_mem.get(5) {
                Ok(_ok) => assert!(false),
                Err(_err) => assert!(true)
            };

            assert_eq!(mut_mem.len(), 5);
            assert_eq!(mut_mem.clone_into_vec(), vec![1, 2, 3, 4, 5]);
        }

        test_new();
        test_from();
        test_from_immut();
    }

    #[test]
    fn test_routine() {
        fn test_new() {
            let mut rtne = Routine::new();

            rtne.push(10);
            
            let pop_ok = rtne.pop();

            assert_eq!(pop_ok.unwrap(), 10);

            let pop_err = rtne.pop();

            assert_eq!(pop_err, None);

            assert_eq!(rtne.len(), 0);
            assert_eq!(rtne.clone_into_vec(), vec![]);
        }

        fn test_from() {
            let mut rtne = Routine::from(vec![]);

            rtne.push(10);
            
            let pop_ok = rtne.pop();

            assert_eq!(pop_ok.unwrap(), 10);

            let pop_err = rtne.pop();

            assert_eq!(pop_err, None);

            assert_eq!(rtne.len(), 0);
            assert_eq!(rtne.clone_into_vec(), vec![]);
        }

        test_new();
        test_from();
    }

    #[test]
    fn test_arch() {
        assert_eq!(WRITE_SCR, 0x57);
    }

    #[test]
    fn test_execute() {
        let rtne = Routine::from(vec![WRITE_SCR, '@' as Byte, WRITE_SCR, 0xA]);
        let result = execute(rtne);

        assert_eq!(result, None)
    }

    #[test]
    fn test_tok() {
        let tok = ir::Token::new(ir::INT_TOKEN, String::from("10"));

        assert_eq!(tok.token_type, ir::INT_TOKEN);
        assert_eq!(tok.token_value, String::from("10"));
    }

    #[test]
    fn test_pos() {
        let mut pos = match ir::Pos::from(String::from("1234\n5678\n90")) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err)
        };

        assert_eq!(pos.chr, '1');
        assert_eq!(pos.pos, 0);
        assert_eq!(pos.lno, 0);
        assert_eq!(pos.onl, 0);

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        assert_eq!(pos.chr, '2');
        assert_eq!(pos.pos, 1);
        assert_eq!(pos.lno, 0);
        assert_eq!(pos.onl, 1);

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        assert_eq!(pos.chr, '5');
        assert_eq!(pos.pos, 5);
        assert_eq!(pos.lno, 1);
        assert_eq!(pos.onl, 1);

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        assert_eq!(pos.chr, '9');
        assert_eq!(pos.pos, 10);
        assert_eq!(pos.lno, 2);
        assert_eq!(pos.onl, 1);

        assert!(pos.has_next());

        match pos.next() {
            Some(some) => panic!("{}", some),
            None => assert!(true)
        };

        assert_eq!(pos.has_next(), true);
    }

    #[test]
    fn test_lexer() {
        let mut lexer = match ir::Lexer::from(String::from("qwe rty uop")) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err)
        };

        let lex_res = match lexer.lex() {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err)
        };

        let expected = vec![
            ir::Token::new(ir::ID_TOKEN, String::from("qwe")),
            ir::Token::new(ir::ID_TOKEN, String::from("rty")),
            ir::Token::new(ir::ID_TOKEN, String::from("uop"))
        ];

        assert_eq!(lex_res.len(), expected.len());

        let mut idx: usize = 0;

        for e in lex_res {
            assert_eq!(e, match expected.get(idx) {
                Some(some) => some.clone(),
                None => ir::Token::new(ir::INT_TOKEN, String::from("1"))
            });
            
            idx = idx + 1;
        }
    }

    #[test]
    fn test_delex() {
        assert!(true);
    }

    #[test]
    fn test_generate() {
        let mut lexer = match ir::Lexer::from("write_scr 10".to_string()) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err)
        };

        let lex_res = match lexer.lex() {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err)
        };

        let res = match ir::generate(lex_res) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err)
        };

        assert_eq!(res, vec![WRITE_SCR, 10]);
    }
}
