use std::fs::File;
use std::io::{Error, Read, Write};

pub const DISK_SIZE: usize = 0xFFFF * 8;

#[derive(Clone)]
pub struct Disk {
    pub data: Box<[u8]>
}

impl Disk {
    pub fn new() -> Disk {
        Disk {
            data: vec![0; DISK_SIZE].into_boxed_slice()
        }
    }

    pub fn from(file: &mut File) -> Result<Disk, Error> {
        let mut disk = Disk::new();
        let mut unboxed: Vec<u8> = Vec::new();
        
        file.read_to_end(&mut unboxed)?;

        let mut i: usize = 0;

        for byte in unboxed {
            disk.data[i] = byte;

            i = i + 1;
        }
        
        Ok(disk)
    }

    pub fn write(&mut self, byte: u8, index: usize) -> Option<String> {
        if index < DISK_SIZE {
            self.data[index] = byte;

            Option::None
        } else {
            Some(format!("index ({}) < DISK_SIZE ({})", index, DISK_SIZE))
        }
    }

    pub fn read(&self, index: usize) -> Result<u8, String> {
        if index < DISK_SIZE {
            Ok(self.data[index])
        } else {
            Err(format!("index ({}) exceeded DISK_SIZE ({})", index, DISK_SIZE))
        }
    }

    pub fn export(&self, file: &mut File) -> Result<usize, Error> {
        file.write(&self.data)?;

        Ok(DISK_SIZE)
    }
}
