use super::*;

#[test]
fn test_io() {
    #[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
    struct TestIO {
        data: Box<[u8]>,
    }

    impl TestIO {
        pub fn new() -> Self {
            TestIO {
                data: vec![0x0; 1 << 30].into_boxed_slice(),
            }
        }
    }

    impl IO for TestIO {
        fn read(&self, addr: usize, byte: &mut u8) {
            *byte = self.data[addr]
        }

        fn write(&mut self, addr: usize, byte: u8) {
            self.data[addr] = byte
        }

        fn capacity(&self) -> usize {
            self.data.len()
        }
    }

    let mut io = TestIO::new();
    let mut byte: u8 = 0;

    io.write(7, 0xA);
    io.read(7, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(io.capacity(), 1 << 30);
}

#[test]
fn test_bus() {
    #[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
    struct TestIO {
        data: Box<[u8]>,
    }

    impl TestIO {
        pub fn new() -> Self {
            TestIO {
                data: vec![0x0; 1 << 30].into_boxed_slice(),
            }
        }
    }

    impl IO for TestIO {
        fn read(&self, addr: usize, byte: &mut u8) {
            *byte = self.data[addr]
        }

        fn write(&mut self, addr: usize, byte: u8) {
            self.data[addr] = byte
        }

        fn capacity(&self) -> usize {
            self.data.len()
        }
    }

    let mut io = TestIO::new();
    let mut bus: Bus<TestIO> = Bus::new(&mut io);
    let mut byte: u8 = 0;

    bus.write(7, 0xA);
    bus.read(7, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(bus.capacity(), 1 << 30);
}

#[test]
fn test_ram() {
    #[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
    struct TestRAMSupport {
        ram: Option<*mut RAM>,
    }

    impl TestRAMSupport {
        pub fn new() -> Self {
            TestRAMSupport { ram: None }
        }
    }

    impl RAMSupport for TestRAMSupport {
        fn connect(&mut self, ram: &mut RAM, port: usize) {
            if port == 0 {
                self.ram = Some(ram);
            }
        }

        fn disconnect(&mut self, port: usize) {
            if port == 0 {
                self.ram = None;
            }
        }

        fn is_connected(&self) -> bool {
            self.ram.is_some()
        }
    }

    let mut ram = RAM::new();
    let mut byte: u8 = 0;

    ram.write(7, 0xA);
    ram.read(7, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(ram.capacity(), 1 << 30);

    ram = RAM::with_capacity(0xFFFF);

    ram.write(7, 0xA);
    ram.read(7, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(ram.capacity(), 0xFFFF);

    let mut ram_sup = TestRAMSupport::new();

    assert_eq!(ram_sup.is_connected(), false);

    ram_sup.connect(&mut ram, 0);

    assert_eq!(ram_sup.is_connected(), true);

    ram_sup.disconnect(0);

    assert_eq!(ram_sup.is_connected(), false);
}

#[test]
fn test_cpu() {
    let mut ram = RAM::new();
    let mut cpu = CPU::new();

    RAMSupport::connect(&mut cpu, &mut ram, 0);
    RAMSupport::disconnect(&mut cpu, 0);
}

#[test]
fn test_storage() {
    use std::fs::{self, File};
    use std::io::Write;

    #[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
    pub struct TestStorageSupport {
        storages: Box<[Option<*mut Storage>]>,
    }

    impl TestStorageSupport {
        pub fn new() -> Self {
            TestStorageSupport {
                storages: vec![None; 4].into_boxed_slice(),
            }
        }
    }

    impl StorageSupport for TestStorageSupport {
        fn connect(&mut self, storage: &mut Storage, port: usize) {
            self.storages[port] = Some(storage);
        }

        fn disconnect(&mut self, port: usize) {
            self.storages[port] = None;
        }

        fn is_connected(&self, port: usize) -> bool {
            self.storages[port].is_some()
        }
    }

    let mut storage = Storage::new();
    let mut byte: u8 = 0;

    storage.write(7, 0xA);
    storage.read(7, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(storage.capacity(), 1 << 30);

    storage = Storage::with_capacity(0xFFFF);

    storage.write(7, 0xA);
    storage.read(7, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(storage.capacity(), 0xFFFF);

    byte = 0;

    let mut file = match File::create("test.bin") {
        Ok(ok) => ok,
        Err(err) => panic!("{}", err),
    };

    match file.write_all(&vec![0x0, 0x1, 0x2, 0x3, 0x4]) {
        Ok(_) => (),
        Err(err) => panic!("{}", err),
    }

    storage = match Storage::from("test.bin") {
        Ok(ok) => ok,
        Err(err) => panic!("{}", err),
    };

    storage.write(4, 0xA);
    storage.read(4, &mut byte);

    assert_eq!(byte, 0xA);
    assert_eq!(storage.capacity(), 5);

    let mut storage_sup = TestStorageSupport::new();

    assert_eq!(storage_sup.is_connected(2), false);

    storage_sup.connect(&mut storage, 2);

    assert_eq!(storage_sup.is_connected(2), true);

    storage_sup.disconnect(2);

    assert_eq!(storage_sup.is_connected(2), false);

    match fs::remove_file("test.bin") {
        Ok(_) => (),
        Err(err) => panic!("{}", err),
    };
}

#[test]
fn test_power() {
    #[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
    struct TestPowerManager {
        power: bool,
    }

    impl TestPowerManager {
        pub fn new() -> Self {
            TestPowerManager { power: false }
        }
    }

    impl PowerManager for TestPowerManager {
        fn power_on(&mut self) {
            if !self.power {
                self.power();

                self.power = true;
            }
        }

        fn power_off(&mut self) {
            if self.power {
                self.power = false;
            }
        }

        fn power(&mut self) {
            if self.power {
                self.power_off();
            } else {
                self.power = true;

                self.power = false;
            }
        }

        fn is_on(&self) -> bool {
            self.power
        }
    }

    let mut power_manager = TestPowerManager::new();

    assert_eq!(power_manager.is_on(), false);

    power_manager.power_on();

    assert_eq!(power_manager.is_on(), true);

    power_manager.power_off();

    assert_eq!(power_manager.is_on(), false);

    power_manager.power();

    assert_eq!(power_manager.is_on(), false);
}
