use super::*;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct CPU {
    power: bool,
    ram_bus: Option<Bus<RAM>>,
    storages: Box<[Option<Bus<Storage>>]>,
    pc: usize,
    sp: usize,
    eax: u32,
    ebx: u32,
    ecx: u32,
    edx: u32,
    ebp: u32,
    esi: u32,
    edi: u32,
    esp: u32,
    ymm0: u256,
    ymm1: u256,
    ymm2: u256,
    ymm3: u256,
    ymm4: u256,
    ymm5: u256,
    ymm6: u256,
    ymm7: u256,
    ymm8: u256,
    ymm9: u256,
    ymm10: u256,
    ymm11: u256,
    ymm12: u256,
    ymm13: u256,
    ymm14: u256,
    ymm15: u256,
    xmm0: u128,
    xmm1: u128,
    xmm2: u128,
    xmm3: u128,
    xmm4: u128,
    xmm5: u128,
    xmm6: u128,
    xmm7: u128,
    xmm8: u128,
    xmm9: u128,
    xmm10: u128,
    xmm11: u128,
    xmm12: u128,
    xmm13: u128,
    xmm14: u128,
    xmm15: u128,
    mmx0: u64,
    mmx1: u64,
    mmx2: u64,
    mmx3: u64,
    mmx4: u64,
    mmx5: u64,
    mmx6: u64,
    mmx7: u64,
    fpr0: u80,
    fpr1: u80,
    fpr2: u80,
    fpr3: u80,
    fpr4: u80,
    fpr5: u80,
    fpr6: u80,
    fpr7: u80,
    eip: u32,
    rip: u64,
    eflags: u32,
    rflags: u64,
}

impl CPU {
    pub fn new() -> Self {
        CPU {
            power: false,
            ram_bus: None,
            storages: vec![None; 8].into_boxed_slice(),
            pc: 0,
            sp: 0,
            eax: 0,
            ebx: 0,
            ecx: 0,
            edx: 0,
            ebp: 0,
            esi: 0,
            edi: 0,
            esp: 0,
            ymm0: u256::new(0),
            ymm1: u256::new(0),
            ymm2: u256::new(0),
            ymm3: u256::new(0),
            ymm4: u256::new(0),
            ymm5: u256::new(0),
            ymm6: u256::new(0),
            ymm7: u256::new(0),
            ymm8: u256::new(0),
            ymm9: u256::new(0),
            ymm10: u256::new(0),
            ymm11: u256::new(0),
            ymm12: u256::new(0),
            ymm13: u256::new(0),
            ymm14: u256::new(0),
            ymm15: u256::new(0),
            xmm0: 0,
            xmm1: 0,
            xmm2: 0,
            xmm3: 0,
            xmm4: 0,
            xmm5: 0,
            xmm6: 0,
            xmm7: 0,
            xmm8: 0,
            xmm9: 0,
            xmm10: 0,
            xmm11: 0,
            xmm12: 0,
            xmm13: 0,
            xmm14: 0,
            xmm15: 0,
            mmx0: 0,
            mmx1: 0,
            mmx2: 0,
            mmx3: 0,
            mmx4: 0,
            mmx5: 0,
            mmx6: 0,
            mmx7: 0,
            fpr0: u80::new(0),
            fpr1: u80::new(0),
            fpr2: u80::new(0),
            fpr3: u80::new(0),
            fpr4: u80::new(0),
            fpr5: u80::new(0),
            fpr6: u80::new(0),
            fpr7: u80::new(0),
            eip: 0,
            rip: 0,
            eflags: 0,
            rflags: 0,
        }
    }
}

impl RAMSupport for CPU {
    fn connect(&mut self, ram: &mut RAM, port: usize) {
        if port == 0 {
            self.ram_bus = Some(Bus::new(ram));
        }
    }

    fn disconnect(&mut self, port: usize) {
        if port == 0 {
            self.ram_bus = None;
        }
    }

    fn is_connected(&self) -> bool {
        self.ram_bus.is_some()
    }
}

impl StorageSupport for CPU {
    fn connect(&mut self, storage: &mut Storage, port: usize) {
        if port < 8 {
            self.storages[port] = Some(Bus::new(storage));
        }
    }

    fn disconnect(&mut self, port: usize) {
        if port < 8 {
            self.storages[port] = None;
        }
    }

    fn is_connected(&self, port: usize) -> bool {
        if port < 8 {
            self.storages[port].is_some()
        } else {
            false
        }
    }
}

impl PowerManager for CPU {
    fn power_on(&mut self) {
        if !self.power {
            self.power = false;
        }
    }

    fn power_off(&mut self) {
        if self.power {
            self.power = false;
        }
    }

    fn power(&mut self) {
        if self.power {
            self.power_off();
        } else {
            self.power_on();

            self.power_off();
        }
    }

    fn is_on(&self) -> bool {
        self.power
    }
}

pub type CPUError = mtk::Error;
