use super::*;

#[test]
fn test_bus() {
    let mut disk = Disk::new(0xFFFF);
    let mut bus = Bus::new(&mut disk);

    match bus.write(0x1, 0xA) {
        Ok(ok) => assert_eq!(ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match bus.read(0x1) {
        Ok(ok) => assert_eq!(*ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match bus.read_mut(0x1) {
        Ok(ok) => *ok = 0xB,
        Err(err) => panic!("{}", err),
    };

    assert_eq!(bus[0x1], 0xB);

    bus[0x1] = 0xC;

    assert_eq!(bus[0x1], 0xC);
    assert_eq!(bus.size(), 0xFFFF);
}

#[test]
fn test_cpu() {
    let mut disk = Disk::new(0xFFFF);
    let mut mem = Mem::new((1 << 30) / 2);
    let mut cpu = CPU::new(&mut mem);

    match cpu.attach_disk(&mut disk, 0) {
        Ok(_) => assert!(true),
        Err(err) => panic!("{}", err),
    };

    match cpu.detach_disk(0) {
        Ok(ok) => assert_eq!(ok, 0xFFFF),
        Err(err) => panic!("{}", err),
    };
}

#[test]
fn test_disk() {
    let mut disk = Disk::new(0xFFFF);

    match disk.write(0x1, 0xA) {
        Ok(ok) => assert_eq!(ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match disk.read(0x1) {
        Ok(ok) => assert_eq!(*ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match disk.read_mut(0x1) {
        Ok(ok) => *ok = 0xB,
        Err(err) => panic!("{}", err),
    };

    assert_eq!(disk.size(), 0xFFFF);
}

#[test]
fn test_io() {
    struct TestIO {
        data: Box<[u8]>,
    }

    impl TestIO {
        fn new() -> Self {
            TestIO {
                data: Box::new([0; 0xFFFF]),
            }
        }
    }

    impl IODev for TestIO {
        fn read(&self, addr: usize) -> Result<&u8, IOError> {
            if addr < self.data.len() {
                Ok(&self.data[addr])
            } else {
                Err(IOError::new("out of bounds"))
            }
        }

        fn read_mut(&mut self, addr: usize) -> Result<&mut u8, IOError> {
            if addr < self.data.len() {
                Ok(&mut self.data[addr])
            } else {
                Err(IOError::new("out of bounds"))
            }
        }

        fn write(&mut self, addr: usize, data: u8) -> Result<usize, IOError> {
            if addr < self.data.len() {
                self.data[addr] = data;

                Ok(data as usize)
            } else {
                Err(IOError::new("out of bounds"))
            }
        }

        fn size(&self) -> usize {
            self.data.len()
        }
    }

    let mut io = TestIO::new();

    match io.write(0x1, 0xA) {
        Ok(ok) => assert_eq!(ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match io.read(0x1) {
        Ok(ok) => assert_eq!(*ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match io.read_mut(0x1) {
        Ok(ok) => *ok = 0xB,
        Err(err) => panic!("{}", err),
    };

    assert_eq!(io.size(), 0xFFFF);
}

#[test]
fn test_mem() {
    let mut mem = Mem::new((1 << 30) / 2);

    match mem.write(0x1, 0xA) {
        Ok(ok) => assert_eq!(ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match mem.read(0x1) {
        Ok(ok) => assert_eq!(*ok, 0xA),
        Err(err) => panic!("{}", err),
    };

    match mem.read_mut(0x1) {
        Ok(ok) => *ok = 0xB,
        Err(err) => panic!("{}", err),
    };

    assert_eq!(mem.size(), (1 << 30) / 2);
}

#[test]
fn test_rom() {
    let rom = ROM::from(&vec![0xA; 0xFFFF]);

    for i in 0..rom.len() {
        match rom.get(i) {
            Some(ok) => assert_eq!(*ok, 0xA),
            None => panic!("out of bounds"),
        };

        assert_eq!(rom[i], 0xA);
    }

    assert_eq!(rom.len(), 0xFFFF);
}

#[test]
fn test_load() {
    use std::fs::{self, File};
    use std::io::Write;

    let mut file = match File::create("test_load.bin") {
        Ok(ok) => ok,
        Err(err) => panic!("{}", err),
    };

    match file.write(&[PUSH, 1, POP]) {
        Ok(ok) => assert_eq!(ok, 3),
        Err(err) => panic!("{}", err),
    };

    let mut rom = load::from(&[PUSH, 1, POP]);

    assert_eq!(rom, ROM::from(&[PUSH, 1, POP]));

    rom = match load::from_file("test_load.bin") {
        Ok(ok) => ok,
        Err(err) => panic!("{}", err),
    };

    assert_eq!(rom, ROM::from(&[PUSH, 1, POP]));

    match fs::remove_file("test_load.bin") {
        Ok(_) => (),
        Err(err) => panic!("{}", err),
    };
}
