use super::*;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Mem {
    inner: Box<[u8]>,
    size: usize,
}

impl Mem {
    pub fn new(size: usize) -> Self {
        Mem {
            inner: vec![0x0; size].into_boxed_slice(),
            size,
        }
    }
}

impl IODev for Mem {
    fn read(&self, addr: usize) -> Result<&u8, IOError> {
        match self.inner.get(addr) {
            Some(some) => Ok(some),
            None => Err(IOError::new(format!("addr ({}) is out of bounds", addr))),
        }
    }

    fn read_mut(&mut self, addr: usize) -> Result<&mut u8, IOError> {
        match self.inner.get_mut(addr) {
            Some(some) => Ok(some),
            None => Err(IOError::new(format!("addr ({}) is out of bounds", addr))),
        }
    }

    fn write(&mut self, addr: usize, data: u8) -> Result<usize, IOError> {
        match self.inner.get_mut(addr) {
            Some(some) => {
                *some = data;

                Ok(data as usize)
            }
            None => Err(IOError::new(format!("addr ({}) is out of bounds", addr))),
        }
    }

    fn size(&self) -> usize {
        self.size
    }
}
