use super::*;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Disk {
    inner: Box<[u8]>,
    size: usize,
}

impl Disk {
    pub fn new(size: usize) -> Self {
        Disk {
            inner: vec![0; size].into_boxed_slice(),
            size,
        }
    }

    pub fn from(bytes: &[u8]) -> Self {
        let mut disk = Disk::new(bytes.len());

        for (i, byte) in bytes.iter().enumerate() {
            disk.inner[i] = *byte;
        }

        disk
    }

    pub fn from_rom(rom: &ROM) -> Self {
        let mut inner: Vec<u8> = Vec::new();

        for i in 0..rom.len() {
            inner.push(match rom.get(i) {
                Some(some) => *some,
                None => 0,
            })
        }

        Disk {
            inner: inner.into_boxed_slice(),
            size: rom.len(),
        }
    }
}

impl IODev for Disk {
    fn read(&self, addr: usize) -> Result<&u8, IOError> {
        match self.inner.get(addr) {
            Some(some) => Ok(some),
            None => Err(IOError::new(format!("addr ({}) is out of bounds", addr))),
        }
    }

    fn read_mut(&mut self, addr: usize) -> Result<&mut u8, IOError> {
        match self.inner.get_mut(addr) {
            Some(some) => Ok(some),
            None => Err(IOError::new(format!("addr ({}) is out of bounds", addr))),
        }
    }

    fn write(&mut self, addr: usize, data: u8) -> Result<usize, IOError> {
        match self.inner.get_mut(addr) {
            Some(some) => {
                *some = data;

                Ok(data as usize)
            }
            None => Err(IOError::new(format!("addr ({}) is out of bounds", addr))),
        }
    }

    fn size(&self) -> usize {
        self.size
    }
}

pub type DiskError = mtk::Error;
