use super::*;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Mem {
    inner: Vec<Option<Box<[Byte]>>>
}

impl Mem {
    pub fn new() -> Mem {
        Mem {
            inner: vec![Some(vec![0x0].into_boxed_slice())]
        }
    }

    pub fn len(&self) -> usize {
        self.inner.len()
    }
    
    pub fn allocate(&mut self, size: usize) -> Result<usize, Error> {
        if size == 0 {
            return Err(Error::from(format!("invalid size `{}`", size)));
        }

        self.inner.push(Some(vec![0x0; size].into_boxed_slice()));

        Ok(self.inner.len() - 1)
    }

    pub fn deallocate(&mut self, addr: usize) -> Result<usize, Error> {
        let chunk = match self.inner.get_mut(addr) {
            Some(some) => some,
            None => return Err(Error::from(format!("invalid address `{}`", addr)))
        };

        let size = match chunk {
            Some(some) => some.len(),
            None => return Err(Error::from(format!("invalid address `{}`", addr)))
        };

        *chunk = None;

        Ok(size)
    }

    pub fn get(&self, addr: usize) -> Option<&Box<[Byte]>> {
        match self.inner.get(addr) {
            Some(some) => {
                match some {
                    Some(some) => Some(some),
                    None => None
                }
            },
            None => None
        }
    }

    pub fn get_mut(&mut self, addr: usize) -> Option<&mut Box<[Byte]>> {
        match self.inner.get_mut(addr) {
            Some(some) => {
                match some {
                    Some(some) => Some(some),
                    None => None
                }
            },
            None => None
        }
    }
    
    pub fn last(&self) -> Option<&Box<[Byte]>> {
        match self.inner.last() {
            Some(some) => {
                match some {
                    Some(some) => Some(some),
                    None => None
                }
            },
            None => None
        }
    }

    pub fn last_mut(&mut self) -> Option<&mut Box<[Byte]>> {
        match self.inner.last_mut() {
            Some(some) => {
                match some {
                    Some(some) => Some(some),
                    None => None
                }
            },
            None => None
        }
    }
}

impl ToString for Mem {
    fn to_string(&self) -> String {
        let mut res = String::new();

        for chunk in self.inner.iter() {
            match chunk {
                Some(chunk) => {
                    for byte in chunk.iter() {
                        res.push_str(&format!("0x{:0X} ", byte));
                    }
                },
                None => {
                    res.push_str("0x0 ");
                }
            }
        }
        
        res
    }
}
