use super::*;

#[test]
fn test_byte() {
    assert_eq!(Byte::MIN, i64::MIN);
    assert_eq!(Byte::MAX, i64::MAX);

    assert_eq!(IPtr::MIN, i64::MIN);
    assert_eq!(IPtr::MAX, i64::MAX);
    
    assert_eq!(RPtr::MIN, usize::MIN);
    assert_eq!(RPtr::MAX, usize::MAX);
}

#[test]
fn test_mem() {
    let mut mem = Mem::new();

    assert_eq!(NULL, 0);
    assert_eq!(mem.size(), 1);

    let addr1 = match mem.alloc(4) {
        Ok(ok) => {
            assert_eq!(ok, 1);
            assert_eq!(mem.size(), 2);

            ok
        },
        Err(err) => panic!("{}", err.to_string())
    };
    
    match mem.get_mut(addr1) {
        Ok(ok) => {
            assert_eq!(ok.len(), 4);

            *match ok.get_mut(0) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            } = 'O' as Byte;

            *match ok.get_mut(1) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            } = 'K' as Byte;

            *match ok.get_mut(2) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            } = 0xA; // '\n' as Byte

            *match ok.get_mut(3) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            } = 0x0; // '\0' as Byte
        },
        Err(err) => panic!("{}", err.to_string())
    };

    match mem.get(addr1) {
        Ok(ok) => {
            assert_eq!(ok.len(), 4);

            assert_eq!(*match ok.get(0) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            }, 'O' as Byte);

            assert_eq!(*match ok.get(1) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            }, 'K' as Byte);

            assert_eq!(*match ok.get(2) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            }, 0xA); // '\n' as Byte

            assert_eq!(*match ok.get(3) {
                Some(ok) => ok,
                None => panic!("{}", "unexpected None")
            }, 0x0); // '\0' as Byte

            assert_eq!(ok.len(), 4);
        },
        Err(err) => panic!("{}", err.to_string())
    };

    match mem.dealloc(addr1) {
        Ok(ok) => {
            assert_eq!(ok, 4);
            assert_eq!(mem.size(), 2);

            ok
        },
        Err(err) => panic!("{}", err.to_string())
    };

    let mut addr2 = raw::malloc(&mut mem, 10);
    assert_eq!(addr2, 2);

    addr2 = raw::realloc(&mut mem, addr2, 20);
    assert_eq!(addr2, 3);

    addr2 = raw::calloc(&mut mem, 3, 10);
    assert_eq!(addr2, 4);
    
    raw::free(&mut mem, addr2);
}

#[test]
fn test_stack() {
    // Assert true because std::vec::Vec is already tested
    assert!(true);
}

#[test]
fn test_io() {
    let mut io_handle = IOHandle::new();

    match io_handle.print(Stdout, "test message") {
        Ok(ok) => {
            assert_eq!(ok, "test message".len());
        },
        Err(err) => panic!("{}", err.to_string())
    };
    
    match io_handle.print(Stderr, "test message") {
        Ok(ok) => {
            assert_eq!(ok, "test message".len());
        },
        Err(err) => panic!("{}", err.to_string())
    };

    match io_handle.print(Stdin, "test message") {
        Ok(_) => panic!("expected error when writing to stdin"),
        Err(_) => assert!(true)
    };

    // Do not test stdin
    assert!(true);
}

#[test]
fn test_exec() {
    let mut exec = Exec::new(&vec![OP_RETURN, 0]);

    assert_eq!(exec.calls().clone(), vec![OP_RETURN, 0]);
    assert_eq!(exec.io().clone(), IOHandle::new());

    let mut mem =  Mem::new();
    let mut execsvc = ExecService::new(&mut mem);

    match unsafe { execsvc.execute(&mut Exec::new(&vec![OP_RETURN, 10]), &vec![]) } {
        Ok(ok) => {
            match ok {
                Ok(ok) => assert_eq!(ok, 10),
                Err(_) => panic!("expected Ok")
            }
        },
        Err(err) => panic!("{}", err.to_string())
    };

    match unsafe { execsvc.execute(&mut Exec::new(&vec![OP_THROW, 20]), &vec![]) } {
        Ok(ok) => {
            match ok {
                Ok(_) => panic!("expected Err"),
                Err(err) => assert_eq!(err, 20)
            }
        },
        Err(err) => panic!("{}", err.to_string())
    };

    match unsafe { execsvc.execute(&mut Exec::new(&vec![
        OP_GOTO, 4,
        OP_RETURN, 30,
        OP_GOTO, 2
    ]), &vec![]) } {
        Ok(ok) => {
            match ok {
                Ok(ok) => assert_eq!(ok, 30),
                Err(_) => panic!("expected Ok")
            }
        },
        Err(err) => panic!("{}", err.to_string())
    };
}

#[test]
fn test_math() {
    assert_eq!(flip(10000), -10000);
    assert_eq!(flip(-10000), 10000);
}
