use super::*;
use mtk::Error;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct Exec {
    calls: Vec<Byte>,
    io_handle: IOHandle
}

impl Exec {
    pub fn new(calls: &Vec<Byte>) -> Exec {
        Exec {
            calls: calls.clone(),
            io_handle: IOHandle::new()
        }
    }

    pub fn io(&mut self) -> &mut IOHandle {
        &mut self.io_handle
    }

    pub fn calls(&mut self) -> &mut Vec<Byte> {
        &mut self.calls
    }
}

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
pub struct ExecService {
    mem: *mut Mem
}

impl ExecService {
    pub fn new(mem: &mut Mem) -> ExecService {
        ExecService { mem }
    }

    pub unsafe fn execute(&mut self, exec: &mut Exec, argv: &Vec<Byte>) -> Result<Result<Byte, Byte>, Error> {
        let calls = exec.calls().clone();
        let mut stdio = exec.io();
        let mut mem = self.mem;
        let mut clock: usize = 0;
        let argv = argv.clone();
        let mut stack = Stack::new();

        while clock != calls.len() {
            if calls[clock] == OP_RETURN {
                let value = match calls.get(clock + 1) {
                    Some(some) => *some,
                    None => return Err(Error::from(format!("invalid address `{}`", clock + 1)))
                };

                return Ok(Ok(value));
            } else if calls[clock] == OP_THROW {
                let value = match calls.get(clock + 1) {
                    Some(some) => *some,
                    None => return Err(Error::from(format!("invalid address `{}`", clock + 1)))
                };

                return Ok(Err(value));
            } else if calls[clock] == OP_JUMP {
                let value = match calls.get(clock + 1) {
                    Some(some) => *some,
                    None => return Err(Error::from(format!("invalid address `{}`", clock + 1)))
                };

                if value < 0 {
                    return Err(Error::from(format!("invalid address `{}`", value)));
                }

                if value as usize >= calls.len() {
                    return Err(Error::from(format!("invalid address `{}`", value)));
                }

                clock = value as usize;
            }
        }

        Ok(Err(NULL))
    }
}
