pub mod env;
pub mod routine;
pub mod arch;

pub use env::*;
pub use routine::*;
pub use arch::*;

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_env() {
        let mut env = Env::new();
        
        let block1 = match env.alloc(10) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        let block2 = match env.alloc(10) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        assert_eq!(block1, 1);
        assert_eq!(block2, 2);
        
        assert_eq!(match env.dealloc(block1) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        }, 10);

        assert_eq!(match env.dealloc(block2) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        }, 10);

        assert_eq!(*env.get_mem(), vec![Some(vec![0x0; 0]), None, None]);
        assert_eq!(*env.get_mem_mut(), vec![Some(vec![0x0; 0]), None, None]);
    
        let routine1 = match env.def(vec![RETURN, 10]) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        let result1 = unsafe { env.call(routine1, vec![]) };
        
        match result1 {
            Ok(ok) => {
                match ok {
                    Ok(ok) => assert_eq!(ok, 10),
                    Err(_err) => panic!("unexpected result")
                }
            },
            Err(err) => panic!("{}", err.to_string())
        };

        let routine2 = match env.def(vec![THROW, 5]) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        let result2 = unsafe { env.call(routine2, vec![]) };
        
        match result2 {
            Ok(ok) => {
                match ok {
                    Ok(_ok) => panic!("unexpected result"),
                    Err(err) => assert_eq!(err, 5)
                }
            },
            Err(err) => panic!("{}", err.to_string())
        };
    }

    #[test]
    fn test_routine() {
        let mut env = Env::new();
        
        let routine2 = match env.def(vec![RETURN, 20]) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };
 
        let mut routine1 = Routine::from(&mut env, vec![RETURN, 10]);

        match unsafe { routine1.call(Vec::new()) } {
            Ok(ok) => {
                match ok {
                    Ok(ok) => assert_eq!(ok, 10),
                    Err(_err) => panic!("unexpected result")
                }
            },
            Err(err) => panic!("{}", err.to_string())
        };

        routine1.push(0x0);

        assert_eq!(match routine1.get(2) {
            Some(some) => *some,
            None => panic!("could not get from address 2")
        }, 0x0);

        assert_eq!(match routine1.pop() {
            Some(some) => some,
            None => panic!("failed to pop")
        }, 0x0);

        *match routine1.get_mut(1) {
            Some(some) => some,
            None => panic!("could not get from address 1")
        } = 20;

        assert_eq!(match routine1.pop() {
            Some(some) => some,
            None => panic!("failed to pop")
        }, 20);

        let mut routine3 = Routine::from(&mut env, vec![CALL, routine2 as i128, 0, RETURN_C]);

        match unsafe { routine3.call(Vec::new()) } {
            Ok(ok) => {
                match ok {
                    Ok(ok) => assert_eq!(ok, 20),
                    Err(_err) => panic!("unexpected result")
                }
            },
            Err(err) => panic!("{}", err.to_string())
        };
    }
}
