pub mod dev;
pub mod arch;
pub mod disk;
pub mod ir;

pub use dev::*;
pub use arch::*;
pub use disk::*;

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_dev() {
        let mut disk = Disk::new();
        let mut device = Device::new();

        match disk.write_all(0, vec![
            JMP_F, 2,
            QHALT,
            JMP_B, 2
        ].into_boxed_slice()) {
            Some(some) => panic!("{}", some.to_string()),
            None => assert!(true)
        };

        device.attach(0, &mut disk);
        device.power();

        device.detach(0);
    }

    #[test]
    fn test_disk() {
        let mut disk = Disk::new();

        match disk.write(0xA, 200) {
            Ok(ok) => assert_eq!(ok, 0x0),
            Err(err) => panic!("{}", err.to_string())
        };

        match disk.read(200) {
            Ok(ok) => assert_eq!(ok, 0xA),
            Err(err) => panic!("{}", err.to_string())
        };

        match disk.write(0x0, 200) {
            Ok(ok) => assert_eq!(ok, 0xA),
            Err(err) => panic!("{}", err.to_string())
        };

        assert_eq!(disk.clone_into_vec(), vec![0x0; 0x3FFF]);
    }

    #[test]
    fn test_token() {
        let token = ir::Token::new(ir::ID_TOKEN, "VALUE".to_string());

        assert_eq!(token.clone_type(), ir::ID_TOKEN);
        assert_eq!(token.clone_value(), "VALUE");
        assert_eq!(token.to_string(), format!("(<ID_TOKEN>, 'VALUE')"));
    }

    #[test]
    fn test_split() {
        assert_eq!(match ir::split("a b c".to_string()) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        }, vec![
            String::from("a"),
            String::from("b"),
            String::from("c")
        ]);
    }

    #[test]
    fn test_lexer() {
        let mut lexer = match ir::Lexer::new(&"abc 123 _i.d@ 789".to_string()) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        assert_eq!(match lexer.res() {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        }, vec![
            ir::Token::new(ir::ID_TOKEN, "abc".to_string()),
            ir::Token::new(ir::INT_TOKEN, "123".to_string()),
            ir::Token::new(ir::ID_TOKEN, "_i.d@".to_string()),
            ir::Token::new(ir::INT_TOKEN, "789".to_string())
        ]);

        assert_eq!(lexer.clone_res(), vec![
            ir::Token::new(ir::ID_TOKEN, "abc".to_string()),
            ir::Token::new(ir::INT_TOKEN, "123".to_string()),
            ir::Token::new(ir::ID_TOKEN, "_i.d@".to_string()),
            ir::Token::new(ir::INT_TOKEN, "789".to_string())
        ]);
    }

    #[test]
    fn test_compile() {
        let mut lexer = match ir::Lexer::new(&"jmp_f 2 qhalt jmp_b 2".to_string()) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        match lexer.res() {
            Ok(_ok) => (),
            Err(err) => panic!("{}", err.to_string())
        };

        let compiled = match ir::compile(&lexer) {
            Ok(ok) => ok,
            Err(err) => panic!("{}", err.to_string())
        };

        assert_eq!(compiled, vec![
            JMP_F, 2,
            QHALT,
            JMP_B, 2
        ]);
    }

    #[test]
    fn test_source() {
        assert!(true);
    }
}
