pub type DiskIOError = mtk::IOError;

pub struct Disk {
    data: Box<[u8]>
}

impl Disk {
    pub fn new() -> Disk {
        Disk {
            data: vec![0x0; 0x3FFF].into_boxed_slice()
        }
    }

    pub fn write(&mut self, byte: u8, addr: usize) -> Result<u8, DiskIOError> {
        if addr < 0x3FFF {
            let old = self.data[addr];
            self.data[addr] = byte;

            Ok(old)
        } else {
            Err(DiskIOError::with_string(format!("addr ({}) exceeds the limit ({})", addr, 0x3FFF)))
        }
    }

    pub fn write_all(&mut self, addr: usize, bytes: Box<[u8]>) -> Option<DiskIOError> {
        if bytes.len() < 0x3FFF {
            for i in 0..bytes.len() {
                self.data[addr + i] = bytes[i];
            }

            None
        } else {
            Some(DiskIOError::with_string(format!("too many bytes ({}) exceeds the limit ({})", bytes.len(), 0x3FFF)))
        }
    }

    pub fn read(&self, addr: usize) -> Result<u8, DiskIOError> {
        if addr < 0x3FFF {
            Ok(self.data[addr])
        } else {
            Err(DiskIOError::with_string(format!("addr ({}) exceeds the limit ({})", addr, 0x3FFF)))
        }
    }

    pub fn clone_into_vec(&self) -> Vec<u8> {
        self.data.to_vec().clone()
    }
}

impl Clone for Disk {
    fn clone(&self) -> Disk {
        Disk {
            data: self.data.clone()
        }
    }
}
