use crate::errors::Error;
use crate::headers::File;

pub mod g2_asymmetric;
pub mod g2_symmetric;

#[derive(Debug, Copy, Clone)]
pub enum G2Mode {
    Asymmetric,
    Symmetric,
}

/// Result from the g2 algorithm
pub struct G2Result {
    pub t: Vec<f64>,
    pub hist: Vec<u64>,
}

/// Parameters for the g2 algorithm
///
/// # Parameters
///    - channel_1: The number of the first input channel into the TCSPC
///    - channel_2: The number of the second input channel into the TCSPC
///    - correlation_window: Length of the correlation window of interest in seconds
///    - resolution: Resolution of the g2 histogram in seconds
#[derive(Debug, Clone)]
pub struct G2Params {
    pub channel_1: i32,
    pub channel_2: i32,
    pub correlation_window: f64,
    pub resolution: f64,
    pub record_ranges: Option<Vec<(usize, usize)>>,
}

/// Computes the second order autocorrelation (g2) between two channels on a TCSPC module.
///
/// ## Parameters
///
/// The parameters to the algorithm are passed via a `G2Params` struct that contains
/// the following:
///    - channel_1: The number of the first input channel into the TCSPC,
///    - channel_2: The number of the second input channel into the TCSPC,
///    - correlation_window: Length of the correlation window of interest in seconds,
///    - resolution: Resolution of the g2 histogram in seconds,
///
/// ## Algorithm description
///
/// The streaming g2 algorithm measures the time difference between a
/// photon arriving at a channel and all the photons that came before it and arrived
/// at the other channel. A histogram of time differences is then built and is the output
/// we are after.
///
/// Computational constraints force us to define beforehand the maximum and minimum
/// time differences, the correlation window, that can fit into the histogram. Having a finite
/// value for the time difference we can store implies that we don't need to measure every single
/// photon arriving at channel A against every previous photon on channel B. Instead, only
/// a finite number of clicks into the past need to be considered.
///
/// Past clicks on each channel are pushed into circular buffers that keep the last N photons
/// that arrived at each of them. A circular buffer allows to always have time ordered arrival
/// times if we look from the head position of the buffer backwards.
/// <img src="https://raw.githubusercontent.com/GCBallesteros/tttr-toolbox/master/images/g2_orderings" alt="second order click orderings" >
///
/// ## Mode
/// The mode determines what version of the g2 algorithm will be used. The symmetric version
/// computes relative delays between channels 1 and 2 and between channels 2 and 1. This
/// makes it possible to access both negative and positive time delays on the g2 historgram.
/// The asymmetric mode only provides positive delays in exchange for a slight performance
/// boost. If you are unsure of what to use go for G2Mode::Symmetric.
///
/// ## Finite buffer artifacts
/// Not being capable to look back to all photons that came before can be a potential
/// source of artifacts on the calculated g2 histograms. At it's most extreme if N=1
/// we only look into the immediately previous photon. This introduces an exponential
/// decay artifact on the g2.
///
/// The maximum size of the correlation window that is artifact free is dependent
/// on the click rate. A quick estimate can be obtained by multiplying the inverse
/// of the click rate times the size of the buffer. E.g. for a buffer of 4096 photons
/// and a click rate of 10e6 Hz we get an artifcat free window of 0.4 milliseconds.
/// Taking into consideration typical emitter lifetimes and collection optics efficiency
/// this should be more than enough to capture any relevant dynamics. If this is
/// not the case for you will need to modify the hard coded maximum buffer size
/// defined on `src/tttr_tools/g2.rs`.
pub fn g2(f: &File, params: &G2Params, mode: G2Mode) -> Result<G2Result, Error> {
    match mode {
        G2Mode::Symmetric => g2_symmetric::g2(f, params),
        G2Mode::Asymmetric => g2_asymmetric::g2(f, params),
    }
}
