//
// TIC TAC TOE
// Author: John Thingstad
// Date: February 2022
//

//------------------------------------------------- UI -------------------------------------

fn to_str(peg: u8) -> String {
    match peg {
        0 => String::from(" "),
        1 => String::from("X"),
        2 => String::from("O"),
        _ => panic!("Illegal peg value!"),
    }
}

fn print_intro() {
    println!("TIC TAC TOE\n");
    println!("Get 3 in a row horizontally, vertically or diagonally to win.\n");
    println!("Me first!\n");
}

fn print_board(board: &[[u8; 3]; 3]) {
    println!("    A   B   C");
    println!("  +---+---+---+");
    println!(
        "1 | {} | {} | {} |",
        to_str(board[0][0]),
        to_str(board[0][1]),
        to_str(board[0][2]),
    );
    println!("  +---+---+---+");
    println!(
        "2 | {} | {} | {} |",
        to_str(board[1][0]),
        to_str(board[1][1]),
        to_str(board[1][2]),
    );
    println!("  +---+---+---+");
    println!(
        "3 | {} | {} | {} |",
        to_str(board[2][0]),
        to_str(board[2][1]),
        to_str(board[2][2]),
    );
    println!("  +---+---+---+");
}

fn query_move() -> [usize; 2] {
    let mut posvec = [0usize; 2];
    let mut tries = 0;
    while tries < 5 {
        tries += 1;

        println!("\nWhat is your move? ");
        let mut move_str = String::new();
        let len = std::io::stdin()
            .read_line(&mut move_str)
            .expect("read_line failed!");
        if len != 3 {
            println!("?");
            continue;
        }
        move_str = move_str.to_uppercase();

        let chars: Vec<&str> = move_str.split("").collect::<Vec<&str>>(); // looks like [ "", "A", "1", ""]
        match chars[1] {
            "A" => {
                posvec[0] = 0;
            }
            "B" => {
                posvec[0] = 1;
            }
            "C" => {
                posvec[0] = 2;
            }
            _ => {
                println!("?");
                continue;
            }
        }
        match chars[2] {
            "1" => {
                posvec[1] = 0;
            }
            "2" => {
                posvec[1] = 1;
            }
            "3" => {
                posvec[1] = 2;
            }
            _ => {
                println!("?");
                continue;
            }
        }
        break;
    }

    if tries == 5 {
        panic!("No answer given.")
    }

    posvec
}

fn posvec_to_move(posvec: &[usize; 2]) -> String {
    let mut posstring = String::new();
    match posvec[0] {
        0 => {
            posstring.push_str("A");
        }
        1 => {
            posstring.push_str("B");
        }
        2 => {
            posstring.push_str("C");
        }
        _ => {
            panic!("Illegal value!");
        }
    }
    match posvec[1] {
        0 => {
            posstring.push_str("1");
        }
        1 => {
            posstring.push_str("2");
        }
        2 => {
            posstring.push_str("3");
        }
        _ => {
            panic!("Illegal value!");
        }
    }
    posstring
}

//--------------------------------- computer moves ------------------------------------

fn decode_candidate(triplet: &[[usize; 2]; 3], the_board: &[[u8; 3]; 3]) -> [u8; 3] {
    // takes a triplet.
    // Which is a diagonal, bidigonal, row or col and reads the peg value at that position
    // returns the three values
    let mut candidate = [0u8; 3];
    let mut i = 0;
    for pos in triplet.iter() {
        candidate[i] = the_board[pos[1]][pos[0]];
        i += 1;
    }
    return candidate;
}

fn decode_result(
    found_match: &[u8; 3],
    triplet: &[[usize; 2]; 3],
    the_board: &[[u8; 3]; 3],
) -> [usize; 2] {
    // you have three peg's in a row in a diagonal, bidiagonal, row or column, called the triplet
    // your solution returns a array of three peg's
    // find the position for the peg that is different
    for i in 0..3 {
        let peg = the_board[triplet[i][1]][triplet[i][0]];
        if peg == found_match[i] {
            continue;
        }
        return triplet[i];
    }
    panic!("found_match: not legal values");
}

fn check_candidate(candidate: [u8; 3]) -> Option<([u8; 3], u8)> {
    // first value of a state row is the current peg values
    // the second is the move to take if you have a match
    // This is irrecspective of wether it is on a diagonal, bidiagonal, row or column
    // They are ordered from best move to worst
    // The priority - if several moves match use the one with the highest priority
    // (which has the lowest value)
    let state = [
        [[1, 1, 0], [1, 1, 1]],
        [[1, 0, 1], [1, 1, 1]],
        [[0, 1, 1], [1, 1, 1]],
        [[2, 2, 0], [2, 2, 1]],
        [[2, 0, 2], [2, 1, 2]],
        [[0, 2, 2], [1, 2, 2]],
        [[1, 0, 0], [1, 1, 0]],
        [[0, 1, 0], [0, 1, 1]],
        [[0, 0, 1], [1, 0, 1]],
        [[2, 0, 0], [2, 1, 0]],
        [[0, 2, 0], [1, 2, 0]],
        [[0, 0, 2], [0, 1, 2]],
        [[0, 0, 0], [0, 1, 0]],
    ];
    let priority = [1, 1, 1, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3];

    let mut i = 0;
    for [pos, mov] in state {
        if pos == candidate {
            return Some((mov, priority[i]));
        }
        i += 1;
    }
    return None;
}

fn find_empty_pos(the_board: &[[u8; 3]; 3]) -> [usize; 2] {
    for j in 0..3 {
        for i in 0..3 {
            if the_board[i][j] == 0 {
                return [j, i];
            }
        }
    }
    panic!("No empty positions on the board!")
}

fn make_move(the_board: &[[u8; 3]; 3]) -> [usize; 2] {
    let triplet_list = [
        // list of triplets of position on the board
        [[0, 0], [1, 1], [2, 2]], // diagonal
        [[0, 2], [1, 1], [2, 0]], // bidiagonal
        [[0, 0], [0, 1], [0, 2]], // coloms
        [[1, 0], [1, 1], [1, 2]],
        [[2, 0], [2, 1], [2, 2]],
        [[0, 0], [1, 0], [2, 0]], // rows
        [[0, 1], [1, 1], [2, 1]],
        [[0, 2], [1, 2], [2, 2]],
    ];
    let mut do_move = [1, 1];
    let mut do_priority = 4;
    for triplet in triplet_list.iter() {
        let candidate = decode_candidate(&triplet, &the_board);

        match check_candidate(candidate) {
            None => continue,
            Some((found_match, priority)) => {
                let candidate_move = decode_result(&found_match, &triplet, &the_board);
                if priority < do_priority {
                    do_priority = priority;
                    do_move = candidate_move;
                }
            }
        }
    }
    if do_priority == 4 {
        do_move = find_empty_pos(&the_board);
    }
    return do_move;
}

//--------------------------------- support functions ------------------------------------------------

fn is_empty(your_move: &[usize; 2], the_board: &[[u8; 3]; 3]) -> bool {
    return the_board[your_move[1]][your_move[0]] == 0;
}

fn game_won(the_board: &[[u8; 3]; 3]) -> bool {
    let triplet_list = [
        // list of triplets of position on the board
        [[0, 0], [1, 1], [2, 2]], // diagonal
        [[0, 2], [1, 1], [2, 0]], // bidiagonal
        [[0, 0], [0, 1], [0, 2]], // coloms
        [[1, 0], [1, 1], [1, 2]],
        [[2, 0], [2, 1], [2, 2]],
        [[0, 0], [1, 0], [2, 0]], // rows
        [[0, 1], [1, 1], [2, 1]],
        [[0, 2], [1, 2], [2, 2]],
    ];
    'triplet_loop: for triplet in triplet_list.iter() {
        let peg = the_board[triplet[0][0]][triplet[0][1]];
        // if board position is blank no three in a row possible
        if peg == 0 {
            continue 'triplet_loop;
        }
        // skip if not three in a row. The same 'X' or 'O' (called peg) as the first
        for pos in triplet.iter() {
            if the_board[pos[0]][pos[1]] != peg {
                continue 'triplet_loop;
            }
        }
        return true;
    }
    return false;
}

fn board_full(the_board: &[[u8; 3]; 3]) -> bool {
    let mut empty_count = 0;

    // board is full is the same as saying it has no empty spaces
    for row in 0..3 {
        for col in 0..3 {
            if the_board[row][col] == 0 {
                empty_count += 1;
            }
        }
    }

    return empty_count == 0;
}

//-------------------------------------------- main game loop ----------------------------------

/// Plays the child game of Tic, Tac, Toe.
///
/// Objective on a 3 x 3 board get thee pegs in a row.
///
/// Computer starts the game.

pub fn play() {
    let mut board = [[0u8; 3]; 3];

    print_intro();

    'game_loop: loop {
        let my_move = make_move(&board);
        board[my_move[1]][my_move[0]] = 1;
        print_board(&board);
        println!("\nMy move was {}", posvec_to_move(&my_move));
        if game_won(&board) {
            print!("\nI won!");
            break 'game_loop;
        }

        if board_full(&board) {
            print!("\nIt's a tie.");
            break 'game_loop;
        }

        'query_loop: loop {
            let your_move = query_move();
            if !is_empty(&your_move, &board) {
                println!("{} is occupied!", posvec_to_move(&your_move));
                continue 'query_loop;
            }

            board[your_move[1]][your_move[0]] = 2;
            print_board(&board);
            if game_won(&board) {
                println!("\nYou won!");
                break 'game_loop;
            }
            break 'query_loop;
        }
        println!("\n-------------------\n")
    }
    println!("\nGAME OVER\n");
}
