/*
MIT License

Copyright (c) 2021 Philipp Schuster

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.
*/

use crate::CRATE_VERSION;
use std::fmt::{Display, Formatter, Write};
use std::net::IpAddr;
use std::time::Duration;

/// The final result of this library. It contains all the measured
/// timings.
#[derive(Debug)]
pub struct TtfbOutcome {
    /// Copy of the user input.
    user_input: String,
    /// The used IP address (resolved by DNS).
    ip_addr: IpAddr,
    /// The port.
    port: u16,
    /// If DNS was required, the duration of this operation.
    dns_duration: Option<Duration>,
    /// Duration of the TCP connection start.
    tcp_connect_duration: Duration,
    /// If https is used, the duration of the TLS handshake.
    tls_handshake_duration: Option<Duration>,
    /// The duration of the HTTP GET request sending.
    http_get_send_duration: Duration,
    /// The duration until the first byte from the HTTP response (the header) was received.
    http_ttfb_duration: Duration,
    // http_content_download_duration: Duration,
}

impl TtfbOutcome {
    pub fn new(
        user_input: String,
        ip_addr: IpAddr,
        port: u16,
        dns_duration: Option<Duration>,
        tcp_connect_duration: Duration,
        tls_handshake_duration: Option<Duration>,
        http_get_send_duration: Duration,
        http_ttfb_duration: Duration,
        // http_content_download_duration: Duration,
    ) -> Self {
        TtfbOutcome {
            user_input,
            ip_addr,
            port,
            dns_duration,
            tcp_connect_duration,
            tls_handshake_duration,
            http_get_send_duration,
            http_ttfb_duration,
            // http_content_download_duration,
        }
    }
    /// Getter for [`Self::user_input`].
    pub fn user_input(&self) -> &str {
        &self.user_input
    }
    /// Getter for [`Self::ip_addr`].
    pub fn ip_addr(&self) -> IpAddr {
        self.ip_addr
    }
    /// Getter for [`Self::port`].
    pub fn port(&self) -> u16 {
        self.port
    }
    /// Getter for [`Self::dns_duration`].
    pub fn dns_duration(&self) -> Option<&Duration> {
        self.dns_duration.as_ref()
    }
    /// Getter for [`Self::tls_handshake_duration`].
    pub fn tls_handshake_duration(&self) -> Option<&Duration> {
        self.tls_handshake_duration.as_ref()
    }
    /// Getter for [`Self::http_ttfb_duration`].
    pub fn http_ttfb_duration(&self) -> Duration {
        self.http_ttfb_duration
    }
    /// Getter for [`Self::tcp_connect_duration`].
    pub fn tcp_connect_duration(&self) -> Duration {
        self.tcp_connect_duration
    }
    /*pub fn http_content_download_duration(&self) -> Duration {
        self.http_content_download_duration
    }*/

    /// Getter for [`Self::http_get_send_duration`].
    pub fn http_get_send_duration(&self) -> Duration {
        self.http_get_send_duration
    }

    /// Returns ms with fractional part from a duration.
    fn to_nice_ms_fraction(dur: &Duration) -> f64 {
        let us = dur.as_micros() as f64;
        let ms = us / 1_000.0;
        ms
    }
}

impl Display for TtfbOutcome {
    fn fmt(&self, f: &mut Formatter<'_>) -> std::fmt::Result {
        // ANSI bold
        f.write_str("\u{1b}[1m").unwrap();
        f.write_fmt(format_args!(
            "TTFB for {} (ttfb@v{})",
            &self.user_input, CRATE_VERSION
        ))?;
        // ANSI reset style
        f.write_str("\u{1b}[0m").unwrap();
        f.write_char('\n')?;
        if let Some(dur) = &self.dns_duration {
            let ms = Self::to_nice_ms_fraction(dur);
            f.write_fmt(format_args!("DNS-Lookup:             {:>7.2}ms", ms))?;
            if ms <= 7.0 {
                f.write_str(" (probably cached)")?;
            }
            f.write_char('\n')?;
        }
        f.write_fmt(format_args!(
            "TCP-Connect:            {:>7.2}ms\n",
            Self::to_nice_ms_fraction(&self.tcp_connect_duration)
        ))?;
        if let Some(dur) = &self.tls_handshake_duration {
            let ms = Self::to_nice_ms_fraction(dur);
            f.write_fmt(format_args!("TLS-Handshake:          {:>7.2}ms\n", ms))?;
        }
        f.write_fmt(format_args!(
            "HTTP-Get-Send:          {:>7.2}ms\n",
            Self::to_nice_ms_fraction(&self.http_get_send_duration)
        ))?;
        f.write_fmt(format_args!(
            "HTTP-TTFB:              {:>7.2}ms",
            Self::to_nice_ms_fraction(&self.http_ttfb_duration)
        ))?;
        /*f.write_fmt(format_args!(
            "HTTP-Content-Download:  {:>7.2}ms",
            Self::to_nice_ms_fraction(&self.http_content_download_duration)
        ))?;*/
        Ok(())
    }
}
