use super::Token;

pub struct Lexer {
    input: String,
    position: usize,
    ch: char
}

impl Lexer {
    pub fn new<S: AsRef<str>>(input: S) -> Lexer {
        Lexer {
            input: input.as_ref().to_string(),
            position: 0,
            ch: ' '
        }
    }

    pub fn read_char(&mut self) {
        if self.position >= self.input.len() {
            self.ch = '\0';
        } else {
            self.ch = self.input.chars().nth(self.position).unwrap();
        }

        self.position += 1;
    }

    pub fn skip_whitespace(&mut self) {
        while self.ch.is_whitespace() {
            self.read_char();
        }
    }

    pub fn read_identifier(&mut self) -> String {
        let position = self.position - 1;

        while "QWERTYUIOPASDFGHJKLZXCVBNMqwertyuopasdfghjklizxcvbnm0123456789._".contains(self.ch) {
            self.read_char();
        }

        self.input[position..self.position - 1].to_string()
    }

    pub fn read_number(&mut self) -> i128 {
        let position = self.position - 1;

        while self.ch.is_numeric() {
            self.read_char();
        }

        let number_string = self.input[position..self.position - 1].to_string();

        let number = match number_string.parse::<i128>() {
            Ok(ok) => ok,
            Err(_) => panic!("could not parse '{}' to i128", number_string)
        };

        number
    }

    pub fn read_string(&mut self) -> String {
        let position = self.position - 1;

        while self.ch != '"' {
            self.read_char();
        }

        self.input[position..self.position - 1].to_string()
    }

    pub fn peek_char(&self) -> char {
        if self.position >= self.input.len() {
            '\0'
        } else {
            self.input.chars().nth(self.position).unwrap()
        }
    }

    pub fn skip_comment(&mut self, terminator: char) {
        while self.ch != terminator {
            self.read_char();
        }
    }

    pub fn read_next_token(&mut self) -> Token {
        self.skip_whitespace();

        let tok = match self.ch {
            '=' => {
                if self.peek_char() == '=' {
                    self.read_char();
                    Token::Equal
                } else {
                    Token::Assign
                }
            },
            '+' => Token::Plus,
            '-' => Token::Minus,
            '!' => {
                if self.peek_char() == '=' {
                    self.read_char();
                    Token::NotEqual
                } else {
                    Token::Bang
                }
            },
            '*' => Token::Asterisk,
            '/' => {
                if self.peek_char() == '/' {
                    self.skip_comment('\n');
                    Token::Comment
                } else {
                    Token::Slash
                }
            },
            '<' => {
                if self.peek_char() == '=' {
                    self.read_char();
                    Token::LessThanEqual
                } else {
                    Token::LessThan
                }
            },
            '>' => {
                if self.peek_char() == '=' {
                    self.read_char();
                    Token::GreaterThanEqual
                } else {
                    Token::GreaterThan
                }
            },
            ';' => Token::Semicolon,
            '(' => Token::LParen,
            ')' => Token::RParen,
            ',' => Token::Comma,
            '{' => Token::LBrace,
            '}' => Token::RBrace,
            '"' => {
                self.read_char();
                Token::String(self.read_string())
            },
            '\0' => Token::EOF,
            _ => {
                if "QWERTYUIOPASDFGHJKLZXCVBNMqwertyuopasdfghjklizxcvbnm._".contains(self.ch) {
                    let id = self.read_identifier();
                    
                    match id.as_str() {
                        "" => Token::Illegal,
                        "if" => Token::If,
                        "else" => Token::Else,
                        "return" => Token::Return,
                        "def" => Token::Def,
                        "throw" => Token::Throw,
                        "call" => Token::Call,
                        _ => Token::Identifier(id)
                    }
                } else if self.ch.is_numeric() {
                    Token::Number(self.read_number())
                } else {
                    Token::Illegal
                }
            }
        };

        self.read_char();

        tok
    }
}
