use clap::{App, AppSettings, SubCommand};
use dialoguer::console::Style;
use dialoguer::{theme::ColorfulTheme, Input};
use std::error::Error;
use trisync::cli;

// Application config

// Create a new package if not exist for the feature (git init)

/// This doc string acts as a help message when the user runs '--help'
/// as do all doc strings on fields

/// ApplicationConfig file
#[tokio::main]
async fn main() -> Result<(), Box<dyn Error>> {
    let matches = App::new("Tririga Sync")
        .version("0.1.3")
        .author("Duy Nghia Tran <duy.nghia.tran1@ibm.com>")
        .about("Synchronize TRIRIGA environment using automated OM deployment")
        .setting(AppSettings::ArgRequiredElseHelp)
        .subcommands(
        vec![
            SubCommand::with_name("init").about("Initialize the project by creating a config.yaml file in the current directory. Note that it is mandatory to setup the directory as a Git directory with upstream configured"),
            SubCommand::with_name("push").about("Export changes from the working TRIRIGA environment and push as a ZIP package to the remote repository"),
            SubCommand::with_name("pull").about("Pull changes from the remote repository and apply to working TRIRIGA environment"),
            SubCommand::with_name("deploy").about("Deploy current package with the last commit ID to working TRIRIGA environment"),
            App::new("object").about("Manage objects included in OM package").setting(AppSettings::ArgRequiredElseHelp).subcommands(
                vec![
                    SubCommand::with_name("add").about("Add an object to OM package"),
                    SubCommand::with_name("delete").about("Delete an object from OM package"),
                ]
            )
        ]
        )
        .get_matches();

    match matches.subcommand() {
        ("init", Some(_)) => cli::init().await?,
        ("pull", Some(_)) => cli::pull().await?,
        ("deploy", Some(_)) => cli::deploy().await?,
        ("push", Some(_)) => {
            let theme = ColorfulTheme {
                values_style: Style::new().yellow().dim(),
                ..ColorfulTheme::default()
            };
            let commit_message: String = Input::with_theme(&theme)
                .with_prompt("Push message")
                .default("Default commit - No message from user".parse().unwrap())
                .interact()?;
            cli::push(&commit_message).await?;
        }
        ("object", Some(object_matches)) => match object_matches.subcommand() {
            ("add", Some(_)) => cli::add_tririga_object_to_config()?,
            ("delete", Some(_)) => cli::delete_tririga_object_from_config()?,
            _ => unreachable!(),
        },
        _ => unreachable!(), // If all subcommands are defined above, anything else is unreachabe!()
    }

    println!("🥂 All done");

    Ok(())
}
