use gtk::prelude::*;
use gtk::subclass::prelude::*;
use gtk::{glib, graphene, gsk};

use once_cell::sync::Lazy;
use std::cell::RefCell;
use std::ops;

use crate::progress_icon::ProgressIconExt;

#[repr(C)]
pub struct TriProgressIconClass {
    pub parent_class: gtk::ffi::GtkWidgetClass,
}

unsafe impl ClassStruct for ProgressIcon {
    type Type = ProgressIcon;
}

impl ops::Deref for TriProgressIconClass {
    type Target = glib::Class<gtk::Widget>;

    fn deref(&self) -> &Self::Target {
        unsafe { &*(self as *const _ as *const Self::Target) }
    }
}

impl ops::DerefMut for TriProgressIconClass {
    fn deref_mut(&mut self) -> &mut glib::Class<gtk::Widget> {
        unsafe { &mut *(self as *mut _ as *mut glib::Class<gtk::Widget>) }
    }
}

#[derive(Debug, Default)]
pub struct ProgressIcon {
    pub progress: RefCell<f32>,
    pub icon_size: RefCell<u32>,
    pub inverted: RefCell<bool>,
    pub clockwise: RefCell<bool>,
}

#[glib::object_subclass]
impl ObjectSubclass for ProgressIcon {
    const NAME: &'static str = "TriProgressIcon";
    type Type = super::ProgressIcon;
    type ParentType = gtk::Widget;

    fn new() -> Self {
        Self {
            progress: RefCell::new(0.0),
            icon_size: RefCell::new(16),
            inverted: RefCell::new(false),
            clockwise: RefCell::new(true),
        }
    }
}

impl ObjectImpl for ProgressIcon {
    fn properties() -> &'static [glib::ParamSpec] {
        static PROPERTIES: Lazy<Vec<glib::ParamSpec>> = Lazy::new(|| {
            vec![
                glib::ParamSpec::new_float(
                    "progress",
                    "Progress",
                    "Progress of the icon",
                    0.0,
                    1.0,
                    0.0,
                    glib::ParamFlags::READWRITE | glib::ParamFlags::EXPLICIT_NOTIFY,
                ),
                glib::ParamSpec::new_uint(
                    "icon-size",
                    "Icon Size",
                    "Size of the icon",
                    0,
                    u32::MAX,
                    16,
                    glib::ParamFlags::READWRITE | glib::ParamFlags::EXPLICIT_NOTIFY,
                ),
                glib::ParamSpec::new_boolean(
                    "inverted",
                    "Inverted",
                    "Invert icon colors",
                    false,
                    glib::ParamFlags::READWRITE | glib::ParamFlags::EXPLICIT_NOTIFY,
                ),
                glib::ParamSpec::new_boolean(
                    "clockwise",
                    "Clockwise",
                    "Direction of the icon",
                    false,
                    glib::ParamFlags::READWRITE | glib::ParamFlags::EXPLICIT_NOTIFY,
                ),
            ]
        });
        PROPERTIES.as_ref()
    }

    fn property(&self, obj: &Self::Type, _id: usize, pspec: &glib::ParamSpec) -> glib::Value {
        match pspec.name() {
            "progress" => obj.progress().to_value(),
            "icon-size" => obj.icon_size().to_value(),
            "inverted" => obj.inverted().to_value(),
            "clockwise" => obj.clockwise().to_value(),
            _ => unreachable!(),
        }
    }

    fn set_property(
        &self,
        obj: &Self::Type,
        _id: usize,
        value: &glib::Value,
        pspec: &glib::ParamSpec,
    ) {
        match pspec.name() {
            "progress" => obj.set_progress(value.get().unwrap()),
            "icon-size" => obj.set_icon_size(value.get().unwrap()),
            "inverted" => obj.set_inverted(value.get().unwrap()),
            "clockwise" => obj.set_clockwise(value.get().unwrap()),
            _ => unreachable!(),
        }
    }

    fn constructed(&self, obj: &Self::Type) {
        self.parent_constructed(obj);

        obj.set_valign(gtk::Align::Center);
    }
}

impl WidgetImpl for ProgressIcon {
    fn snapshot(&self, widget: &Self::Type, snapshot: &gtk::Snapshot) {
        let size = widget.icon_size() as f32;
        let radius = size / 2.0;
        let mut color = widget.style_context().color();
        let progress = if widget.clockwise() {
            1.0 - widget.progress()
        } else {
            widget.progress()
        };

        let rect = graphene::Rect::new(0.0, 0.0, size, size);
        let circle = gsk::RoundedRect::from_rect(rect.clone(), radius);
        let center = graphene::Point::new(size / 2.0, size / 2.0);

        if widget.inverted() {
            color.alpha = 1.0;
        } else {
            color.alpha = 0.15;
        }
        let color_stop = gsk::ColorStop::new(progress, color);

        if widget.inverted() {
            color.alpha = 0.15;
        } else {
            color.alpha = 1.0;
        }
        let color_stop_end = gsk::ColorStop::new(progress, color);

        let rotation = 0.0;
        snapshot.push_rounded_clip(&circle);
        snapshot.append_conic_gradient(&rect, &center, rotation, &[color_stop, color_stop_end]);
        snapshot.pop();
    }

    fn measure(
        &self,
        widget: &Self::Type,
        _orientation: gtk::Orientation,
        _for_size: i32,
    ) -> (i32, i32, i32, i32) {
        (widget.icon_size() as i32, widget.icon_size() as i32, -1, -1)
    }
}
