<p align=center>
  <img src="https://raw.githubusercontent.com/cncf/artwork/master/projects/tremor/horizontal/color/tremor-horizontal-color.png" width='35%'> 
  <img src="https://raw.githubusercontent.com/cncf/artwork/master/projects/opentelemetry/horizontal/color/opentelemetry-horizontal-color.png" width='35%'>
</p>
<p align=center><a href="https://landscape.cncf.io/selected=tremor">CNCF Early Stage Sandbox Project</p>
<p align=center><a href="https://landscape.cncf.io/category=streaming-messaging&format=card-mode&grouping=category">CNCF Streaming &amp; Messaging</p>

<hr>

[![Quality Checks]][actions-Checks]
[![License Checks]][actions-License-audit]
[![Security Checks]][actions-Security-audit]
[![CII Best Practices]][bestpractices]
[![GitHub]](LICENSE)

[Quality Checks]: https://github.com/tremor-rs/tremor-otelapis/workflows/Checks/badge.svg
[actions-Checks]: https://github.com/tremor-rs/tremor-otelapis/actions?query=workflow%3AChecks
[License Checks]: https://github.com/tremor-rs/tremor-otelapis/workflows/License%20audit/badge.svg
[actions-License-audit]: https://github.com/tremor-rs/tremor-otelapis/actions?query=workflow%3A%22License+audit%22
[Security Checks]: https://github.com/tremor-rs/tremor-otelapis/workflows/Security%20audit/badge.svg
[actions-Security-audit]: https://github.com/tremor-rs/tremor-otelapis/actions?query=workflow%3A%22Security+audit%22
[CII Best Practices]: https://bestpractices.coreinfrastructure.org/projects/4356/badge
[bestpractices]: https://bestpractices.coreinfrastructure.org/projects/4356
[GitHub]: https://img.shields.io/github/license/tremor-rs/tremor-otelapis

# OpenTelemetry support for Tremor

## Status

Exprimental - this library is a work in progress under active development

## About

A `tonic-build` based data-binding and utility code for OpenTelemetry v1
primarily for use in the [tremor](https://www.tremor.rs/) project for
OpenTelemetry v1 interoperability, integration and interworking.


---


The code in the `gen` folder was seeded by [`tonic-build`].


The code in the `src` folder extends the generated source with utility
code to allow for convenient usage and definition of tonic-based gRPC
servers. Specifically, this library is designed for use by the Tremor Project
but has no dependencies on tremor and can be used standalone.

This library does not provide an API or SDK designed for use as a tracing
facility. The official [OpenTelemetry Rust](https://github.com/open-telemetry/opentelemetry-rust) project
is a complete OpenTelemetry SDK designed for that purpose. It uses the same
underlying protocol buffer definitions and will be a better target for projects that
require OpenTelemetry based observability instrumentation and iteroperability with
the wider observability ecosystem through third party crates.

This library is designed for system integration and interoperability and is not
recommended for use as a tracing SDK or for instrumentation as that is well covered
already by the OpenTelemetry Rust crate. For instrumentation, use the official crate.

For those projects that need basic interworking, interoperability or integration with
OpenTelemetry based systems at a wire level, this project may be useful.

Minimal dependencies for `Cargo.toml`


```toml
[dependencies]
tremor-otelapis = { version = "0.1", features = ["otel-all"] }
tonic = { version = "0.4", features = ["tls"] }
prost = "0.7"
prost-types = "0.7"
tokio = { version = "1.1", features = ["rt-multi-thread", "time", "fs", "macros"] }
```

---

## Example OpenTelemetry Log client. 


Note that clients simply use the generated client stub code from `tonic-build`. 
This library adds no extra utility or convenience.

```rust
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    let channel = Endpoint::from_static("http://0.0.0.0:4316")
        .connect()
        .await?;

    let mut client = LogsServiceClient::new(channel);

    let resource_logs = ResourceLogs {
        ...
    };

    client
        .export(ExportLogsServiceRequest {
            resource_logs: vec![resource_logs],
        })
        .await?;

    Ok(())
}
```

---


## Example OpenTelemetry Log Server.

Note that we use utility code to expose the server side functionality. We pass through the generated
Protocol Buffer message data binding generated code unadorned. The
data bindings for protocol buffer messages are generated by `tonic-build`.

The `tonic-build` in turns builds on `prost-build`.

```rust
fn on_logs(
    request: tonic::Request<ExportLogsServiceRequest>,
) -> Result<tonic::Response<ExportLogsServiceResponse>, tonic::Status> {
    println!("Got a request from {:?}", request.remote_addr());
    let reply = ExportLogsServiceResponse::default();
    Ok(tonic::Response::new(reply))
}

#[tokio::main]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    let addr = "0.0.0.0:4317".parse()?;
    let svc = otelapis::logs::make_service(Box::new(on_logs));
    Server::builder().add_service(svc).serve(addr).await?;

    Ok(())
}

```

## Example async-channel based OpenTelemetry server

For ease of integration with async runtimes such as [tremor](https://www.tremor.rs):

```rust
#[tokio::main]
async fn main() -> Result<(), Box<dyn std::error::Error>> {
    let addr = "0.0.0.0:4317".parse()?;
    let (tx, rx) = bounded(128);
    tremor_otelapis::all::make(addr, tx).await?;

    // ...

    loop {
        match rx.try_recv() {
            Ok(OpenTelemetryEvents::Metrics(metrics)) => {
                // Do something with metrics request
            }
            Ok(OpenTelemetryEvents::Logs(log)) => {
                // Do something with log request
            }
            Ok(OpenTelemetryEvents::Trace(trace)) => {
                // Do something with trace request
            }
            _ => error!("Unsupported"),
        };
   }

   // ...
}
```

[`otelapis`]: https://github.com/open-telemetry/opentelemetry-specification
[`tonic-build`]: https://github.com/hyperium/tonic/tree/master/tonic-build

