#[allow(dead_code)]
pub const SEPARATORS: &[char] = &[
    '\u{0020}', '\u{00A0}', '\u{1680}', '\u{180E}', '\u{2000}', '\u{2001}', '\u{2002}', '\u{2003}',
    '\u{2004}', '\u{2005}', '\u{2006}', '\u{2007}', '\u{2008}', '\u{2009}', '\u{200A}', '\u{200B}',
    '\u{202F}', '\u{205F}', '\u{3000}', '\u{FEFF}',
];

#[derive(PartialEq)]
pub enum Case {
    Upper,
    Mixed,
    Lower,
}

#[derive(PartialEq)]
pub struct Character<'a> {
    pub value: &'a [char],
    pub case: Case,
}

macro_rules! char {
    ($($c:literal),+, upper) => {
        Character {
            value: &[$($c,)+],
            case: Case::Upper,
        }
    };
    ($($c:literal),+, mixed) => {
        Character {
            value: &[$($c,)+],
            case: Case::Mixed,
        }
    };
    ($($c:literal),+, lower) => {
        Character {
            value: &[$($c,)+],
            case: Case::Lower,
        }
    };
}

pub const EMPTY: &[Character] = &[];

// Borrowed from Jovan Turanjanin's Ćirilizator
// https://github.com/turanjanin/cirilizator

pub const LATIN_CLEAN: &[Character] = &[
    char!['A', upper],
    char!['B', upper],
    char!['V', upper],
    char!['G', upper],
    char!['D', upper],
    char!['Đ', upper],
    char!['E', upper],
    char!['Ž', upper],
    char!['Z', upper],
    char!['I', upper],
    char!['J', upper],
    char!['K', upper],
    char!['L', upper],
    char!['L', 'J', upper], //
    char!['L', 'j', mixed], //
    char!['M', upper],
    char!['N', upper],
    char!['N', 'J', upper], //
    char!['N', 'j', mixed], //
    char!['O', upper],
    char!['P', upper],
    char!['R', upper],
    char!['S', upper],
    char!['T', upper],
    char!['Ć', upper],
    char!['U', upper],
    char!['F', upper],
    char!['H', upper],
    char!['C', upper],
    char!['Č', upper],
    char!['D', 'Ž', upper], //
    char!['D', 'ž', mixed], //
    char!['Š', upper],
    char!['a', lower],
    char!['b', lower],
    char!['v', lower],
    char!['g', lower],
    char!['d', lower],
    char!['đ', lower],
    char!['e', lower],
    char!['ž', lower],
    char!['z', lower],
    char!['i', lower],
    char!['j', lower],
    char!['k', lower],
    char!['l', lower],
    char!['l', 'j', lower],
    char!['m', lower],
    char!['n', lower],
    char!['n', 'j', lower],
    char!['o', lower],
    char!['p', lower],
    char!['r', lower],
    char!['s', lower],
    char!['t', lower],
    char!['ć', lower],
    char!['u', lower],
    char!['f', lower],
    char!['h', lower],
    char!['c', lower],
    char!['č', lower],
    char!['d', 'ž', lower],
    char!['š', lower],
];

pub const LATIN_CLEAN_UNICODE: &[Character] = &[
    char!['A', upper],
    char!['B', upper],
    char!['V', upper],
    char!['G', upper],
    char!['D', upper],
    char!['Đ', upper],
    char!['E', upper],
    char!['Ž', upper],
    char!['Z', upper],
    char!['I', upper],
    char!['J', upper],
    char!['K', upper],
    char!['L', upper],
    char!['Ǉ', upper], //
    char!['ǈ', mixed], //
    char!['M', upper],
    char!['N', upper],
    char!['Ǌ', upper], //
    char!['ǋ', mixed], //
    char!['O', upper],
    char!['P', upper],
    char!['R', upper],
    char!['S', upper],
    char!['T', upper],
    char!['Ć', upper],
    char!['U', upper],
    char!['F', upper],
    char!['H', upper],
    char!['C', upper],
    char!['Č', upper],
    char!['Ǆ', upper], //
    char!['ǅ', mixed], //
    char!['Š', upper],
    char!['a', lower],
    char!['b', lower],
    char!['v', lower],
    char!['g', lower],
    char!['d', lower],
    char!['đ', lower],
    char!['e', lower],
    char!['ž', lower],
    char!['z', lower],
    char!['i', lower],
    char!['j', lower],
    char!['k', lower],
    char!['l', lower],
    char!['ǉ', lower],
    char!['m', lower],
    char!['n', lower],
    char!['ǌ', lower],
    char!['o', lower],
    char!['p', lower],
    char!['r', lower],
    char!['s', lower],
    char!['t', lower],
    char!['ć', lower],
    char!['u', lower],
    char!['f', lower],
    char!['h', lower],
    char!['c', lower],
    char!['č', lower],
    char!['ǆ', lower],
    char!['š', lower],
];

pub const CYRILLIC_CLEAN: &[Character] = &[
    char!['А', upper],
    char!['Б', upper],
    char!['В', upper],
    char!['Г', upper],
    char!['Д', upper],
    char!['Ђ', upper],
    char!['Е', upper],
    char!['Ж', upper],
    char!['З', upper],
    char!['И', upper],
    char!['Ј', upper],
    char!['К', upper],
    char!['Л', upper],
    char!['Љ', upper], //
    char!['Љ', mixed], //
    char!['М', upper],
    char!['Н', upper],
    char!['Њ', upper], //
    char!['Њ', mixed], //
    char!['О', upper],
    char!['П', upper],
    char!['Р', upper],
    char!['С', upper],
    char!['Т', upper],
    char!['Ћ', upper],
    char!['У', upper],
    char!['Ф', upper],
    char!['Х', upper],
    char!['Ц', upper],
    char!['Ч', upper],
    char!['Џ', upper], //
    char!['Џ', mixed], //
    char!['Ш', upper],
    char!['а', lower],
    char!['б', lower],
    char!['в', lower],
    char!['г', lower],
    char!['д', lower],
    char!['ђ', lower],
    char!['е', lower],
    char!['ж', lower],
    char!['з', lower],
    char!['и', lower],
    char!['ј', lower],
    char!['к', lower],
    char!['л', lower],
    char!['љ', lower],
    char!['м', lower],
    char!['н', lower],
    char!['њ', lower],
    char!['о', lower],
    char!['п', lower],
    char!['р', lower],
    char!['с', lower],
    char!['т', lower],
    char!['ћ', lower],
    char!['у', lower],
    char!['ф', lower],
    char!['х', lower],
    char!['ц', lower],
    char!['ч', lower],
    char!['џ', lower],
    char!['ш', lower],
];

pub const LATIN_DIRTY: &[Character] = &[
    char!['A', upper],
    char!['B', upper],
    char!['V', upper],
    char!['G', upper],
    char!['D', upper],
    char!['Đ', upper],
    char!['Ð', upper],
    char!['D', 'J', upper],
    char!['D', 'j', mixed],
    char!['E', upper],
    char!['Z', upper],
    char!['Ž', upper],
    char!['\u{17d}', upper],
    char!['I', upper],
    char!['J', upper],
    char!['K', upper],
    char!['L', upper],
    char!['L', 'J', upper],
    char!['Ǉ', upper],
    char!['L', 'j', mixed],
    char!['ǈ', mixed],
    char!['M', upper],
    char!['N', upper],
    char!['N', 'J', upper],
    char!['Ǌ', upper],
    char!['N', 'j', mixed],
    char!['ǋ', mixed],
    char!['O', upper],
    char!['P', upper],
    char!['R', upper],
    char!['S', upper],
    char!['T', upper],
    char!['Ć', upper],
    char!['\u{106}', upper],
    char!['U', upper],
    char!['F', upper],
    char!['H', upper],
    char!['C', upper],
    char!['Č', upper],
    char!['\u{10c}', upper],
    char!['D', 'Ž', upper],
    char!['Ǆ', upper],
    char!['D', '\u{17d}', upper],
    char!['D', 'ž', mixed],
    char!['ǅ', mixed],
    char!['D', '\u{17e}', mixed],
    char!['Š', upper],
    char!['\u{160}', upper],
    char!['a', lower],
    char!['æ', lower],
    char!['b', lower],
    char!['v', lower],
    char!['g', lower],
    char!['d', lower],
    char!['đ', lower],
    char!['d', 'j', lower],
    char!['e', lower],
    char!['z', lower],
    char!['ž', lower],
    char!['\u{17e}', lower],
    char!['i', lower],
    char!['ĳ', lower],
    char!['j', lower],
    char!['k', lower],
    char!['l', lower],
    char!['l', 'j', lower],
    char!['ǉ', lower],
    char!['m', lower],
    char!['n', lower],
    char!['n', 'j', lower],
    char!['ǌ', lower],
    char!['o', lower],
    char!['œ', lower],
    char!['p', lower],
    char!['r', lower],
    char!['s', lower],
    char!['ﬆ', lower],
    char!['t', lower],
    char!['ć', lower],
    char!['\u{107}', lower],
    char!['u', lower],
    char!['f', lower],
    char!['ﬁ', lower],
    char!['ﬂ', lower],
    char!['h', lower],
    char!['c', lower],
    char!['č', lower],
    char!['\u{10d}', lower],
    char!['d', 'ž', lower],
    char!['ǆ', lower],
    char!['d', '\u{17e}', lower],
    char!['š', lower],
    char!['\u{161}', lower],
];

pub const LATIN_DIRTY_UNICODE: &[Character] = &[
    char!['A', upper],
    char!['B', upper],
    char!['V', upper],
    char!['G', upper],
    char!['D', upper],
    char!['Đ', upper],
    char!['Đ', upper],
    char!['Đ', upper],
    char!['Đ', upper],
    char!['E', upper],
    char!['Z', upper],
    char!['Ž', upper],
    char!['Ž', upper],
    char!['I', upper],
    char!['J', upper],
    char!['K', upper],
    char!['L', upper],
    char!['Ǉ', upper],
    char!['Ǉ', upper],
    char!['ǈ', mixed],
    char!['ǈ', mixed],
    char!['M', upper],
    char!['N', upper],
    char!['Ǌ', upper],
    char!['Ǌ', upper],
    char!['ǋ', mixed],
    char!['ǋ', mixed],
    char!['O', upper],
    char!['P', upper],
    char!['R', upper],
    char!['S', upper],
    char!['T', upper],
    char!['Ć', upper],
    char!['Ć', upper],
    char!['U', upper],
    char!['F', upper],
    char!['H', upper],
    char!['C', upper],
    char!['Č', upper],
    char!['Č', upper],
    char!['Ǆ', upper],
    char!['Ǆ', upper],
    char!['Ǆ', upper],
    char!['ǅ', mixed],
    char!['ǅ', mixed],
    char!['ǅ', mixed],
    char!['Š', upper],
    char!['Š', upper],
    char!['a', lower],
    char!['a', 'e', lower],
    char!['b', lower],
    char!['v', lower],
    char!['g', lower],
    char!['d', lower],
    char!['đ', lower],
    char!['đ', lower],
    char!['e', lower],
    char!['z', lower],
    char!['ž', lower],
    char!['ž', lower],
    char!['i', lower],
    char!['i', 'j', lower],
    char!['j', lower],
    char!['k', lower],
    char!['l', lower],
    char!['ǉ', lower],
    char!['ǉ', lower],
    char!['m', lower],
    char!['n', lower],
    char!['ǌ', lower],
    char!['ǌ', lower],
    char!['o', lower],
    char!['o', 'e', lower],
    char!['p', lower],
    char!['r', lower],
    char!['s', lower],
    char!['š', 't', lower],
    char!['t', lower],
    char!['ć', lower],
    char!['ć', lower],
    char!['u', lower],
    char!['f', lower],
    char!['f', 'i', lower],
    char!['f', 'l', lower],
    char!['h', lower],
    char!['c', lower],
    char!['č', lower],
    char!['č', lower],
    char!['ǆ', lower],
    char!['ǆ', lower],
    char!['ǆ', lower],
    char!['š', lower],
    char!['š', lower],
];

pub const CYRILLIC_DIRTY: &[Character] = &[
    char!['А', upper],
    char!['Б', upper],
    char!['В', upper],
    char!['Г', upper],
    char!['Д', upper],
    char!['Ђ', upper],
    char!['Ђ', upper],
    char!['Ђ', upper],
    char!['Ђ', upper],
    char!['Е', upper],
    char!['З', upper],
    char!['Ж', upper],
    char!['Ж', upper],
    char!['И', upper],
    char!['Ј', upper],
    char!['К', upper],
    char!['Л', upper],
    char!['Љ', upper],
    char!['Љ', upper],
    char!['Љ', mixed],
    char!['Љ', mixed],
    char!['М', upper],
    char!['Н', upper],
    char!['Њ', upper],
    char!['Њ', upper],
    char!['Њ', mixed],
    char!['Њ', mixed],
    char!['О', upper],
    char!['П', upper],
    char!['Р', upper],
    char!['С', upper],
    char!['Т', upper],
    char!['Ћ', upper],
    char!['Ћ', upper],
    char!['У', upper],
    char!['Ф', upper],
    char!['Х', upper],
    char!['Ц', upper],
    char!['Ч', upper],
    char!['Ч', upper],
    char!['Џ', upper],
    char!['Џ', upper],
    char!['Џ', upper],
    char!['Џ', mixed],
    char!['Џ', mixed],
    char!['Џ', mixed],
    char!['Ш', upper],
    char!['Ш', upper],
    char!['а', lower],
    char!['а', 'е', lower],
    char!['б', lower],
    char!['в', lower],
    char!['г', lower],
    char!['д', lower],
    char!['ђ', lower],
    char!['ђ', lower],
    char!['е', lower],
    char!['з', lower],
    char!['ж', lower],
    char!['ж', lower],
    char!['и', lower],
    char!['и', 'ј', lower],
    char!['ј', lower],
    char!['к', lower],
    char!['л', lower],
    char!['љ', lower],
    char!['љ', lower],
    char!['м', lower],
    char!['н', lower],
    char!['њ', lower],
    char!['њ', lower],
    char!['о', lower],
    char!['о', 'е', lower],
    char!['п', lower],
    char!['р', lower],
    char!['с', lower],
    char!['с', 'т', lower],
    char!['т', lower],
    char!['ћ', lower],
    char!['ћ', lower],
    char!['у', lower],
    char!['ф', lower],
    char!['ф', 'и', lower],
    char!['ф', 'л', lower],
    char!['х', lower],
    char!['ц', lower],
    char!['ч', lower],
    char!['ч', lower],
    char!['џ', lower],
    char!['џ', lower],
    char!['џ', lower],
    char!['ш', lower],
    char!['ш', lower],
];

pub struct DigraphException<'a> {
    pub latin: &'a [Character<'a>],
    pub latinized: &'a [Character<'a>],
    pub cyrillic: &'a [Character<'a>],
    pub exceptions: &'a [&'a str],
}

pub const DIGRAPH_EXCEPTIONS: &[DigraphException<'static>] = &[
    DigraphException {
        latin: &[
            char!['Đ', upper],
            char!['Ð', upper],
            char!['D', 'J', upper],
            char!['D', 'j', mixed],
            char!['đ', lower],
            char!['d', 'j', lower],
        ],
        latinized: &[
            char!['D', 'J', upper],
            char!['D', 'J', upper],
            char!['D', 'J', upper],
            char!['D', 'j', mixed],
            char!['d', 'j', lower],
            char!['d', 'j', lower],
        ],
        cyrillic: &[
            char!['Д', 'J', upper],
            char!['Д', 'J', upper],
            char!['Д', 'Ј', upper],
            char!['Д', 'ј', mixed],
            char!['д', 'ј', lower],
            char!['д', 'ј', lower],
        ],
        exceptions: DIGRAPH_EXCEPTIONS_DJ,
    },
    DigraphException {
        latin: &[
            char!['D', 'Ž', upper],
            char!['Ǆ', upper],
            char!['D', '\u{17d}', upper],
            char!['D', 'ž', mixed],
            char!['ǅ', mixed],
            char!['D', '\u{17e}', mixed],
            char!['d', 'ž', lower],
            char!['ǆ', lower],
        ],
        latinized: &[
            char!['D', 'Ž', upper],
            char!['D', 'Ž', upper],
            char!['D', 'Ž', upper],
            char!['D', 'ž', mixed],
            char!['D', 'ž', mixed],
            char!['D', 'ž', mixed],
            char!['d', 'ž', lower],
            char!['d', 'ž', lower],
        ],
        cyrillic: &[
            char!['Д', 'Ж', upper],
            char!['Д', 'Ж', upper],
            char!['Д', 'Ж', upper],
            char!['Д', 'ж', mixed],
            char!['Д', 'ж', mixed],
            char!['Д', 'ж', mixed],
            char!['д', 'ж', lower],
            char!['д', 'ж', lower],
        ],
        exceptions: DIGRAPH_EXCEPTIONS_DZ,
    },
    DigraphException {
        latin: &[
            char!['N', 'J', upper],
            char!['Ǌ', upper],
            char!['N', 'j', mixed],
            char!['ǋ', mixed],
            char!['n', 'j', lower],
            char!['ǌ', lower],
        ],
        latinized: &[
            char!['N', 'J', upper],
            char!['N', 'J', upper],
            char!['N', 'j', mixed],
            char!['N', 'j', mixed],
            char!['n', 'j', lower],
            char!['n', 'j', lower],
        ],
        cyrillic: &[
            char!['Н', 'Ј', upper],
            char!['Н', 'Ј', upper],
            char!['Н', 'ј', mixed],
            char!['Н', 'ј', mixed],
            char!['н', 'ј', lower],
            char!['н', 'ј', lower],
        ],
        exceptions: DIGRAPH_EXCEPTIONS_NJ,
    },
];

pub const DIGRAPH_EXCEPTIONS_DJ: &[&str] = &[
    "adjektiv",
    "adjunkt",
    "bazdje",
    "bdje",
    "bezdje",
    "blijedje",
    "bludje",
    "bridjе",
    "vidjel",
    "vidjet",
    "vindjakn",
    "višenedje",
    "vrijedje",
    "gdje",
    "gudje",
    "gdjir",
    "daždje",
    "dvonedje",
    "devetonedje",
    "desetonedje",
    "djb",
    "djeva",
    "djevi",
    "djevo",
    "djed",
    "djejstv",
    "djel",
    "djenem",
    "djeneš",
    "djenu",
    "djet",
    "djec",
    "dječ",
    "djuar",
    "djubison",
    "djubouz",
    "djuer",
    "djui",
    "djuks",
    "djulej",
    "djumars",
    "djupont",
    "djurant",
    "djusenberi",
    "djuharst",
    "djuherst",
    "dovdje",
    "dogrdje",
    "dodjel",
    "drvodje",
    "drugdje",
    "elektrosnabdje",
    "žudje",
    "zabludje",
    "zavidje",
    "zavrijedje",
    "zagudje",
    "zadjev",
    "zadjen",
    "zalebdje",
    "zaludje",
    "zaodje",
    "zapodje",
    "zarudje",
    "zasjedje",
    "zasmrdje",
    "zastidje",
    "zaštedje",
    "zdje",
    "zlodje",
    "igdje",
    "izbledje",
    "izblijedje",
    "izvidje",
    "izdjejst",
    "izdjelj",
    "izludje",
    "isprdje",
    "jednonedje",
    "kojegdje",
    "kudjelj",
    "lebdje",
    "ludjel",
    "ludjet",
    "makfadjen",
    "marmadjuk",
    "međudjel",
    "nadjaha",
    "nadjača",
    "nadjeb",
    "nadjev",
    "nadjenul",
    "nadjenuo",
    "nadjenut",
    "negdje",
    "nedjel",
    "nadjunač",
    "nenadjača",
    "nenadjebi",
    "nenavidje",
    "neodje",
    "nepodjarm",
    "nerazdje",
    "nigdje",
    "obdjel",
    "obnevidje",
    "ovdje",
    "odjav",
    "odjah",
    "odjaš",
    "odjeb",
    "odjev",
    "odjed",
    "odjezd",
    "odjek",
    "odjel",
    "odjen",
    "odjeć",
    "odjec",
    "odjur",
    "odsjedje",
    "ondje",
    "opredje",
    "osijedje",
    "osmonedje",
    "pardju",
    "perdju",
    "petonedje",
    "poblijedje",
    "povidje",
    "pogdjegdje",
    "pogdje",
    "podjakn",
    "podjamč",
    "podjastu",
    "podjemč",
    "podjar",
    "podjeb",
    "podjed",
    "podjezič",
    "podjel",
    "podjen",
    "podjet",
    "pododjel",
    "pozavidje",
    "poludje",
    "poljodjel",
    "ponegdje",
    "ponedjelj",
    "porazdje",
    "posijedje",
    "posjedje",
    "postidje",
    "potpodjel",
    "poštedje",
    "pradjed",
    "prdje",
    "preblijedje",
    "previdje",
    "predvidje",
    "predjel",
    "preodjen",
    "preraspodje",
    "presjedje",
    "pridjev",
    "pridjen",
    "prismrdje",
    "prištedje",
    "probdje",
    "problijedje",
    "prodjen",
    "prolebdje",
    "prosijedje",
    "prosjedje",
    "protivdjel",
    "prošlonedje",
    "radjard",
    "razvidje",
    "razdjev",
    "razdjel",
    "razodje",
    "raspodje",
    "rasprdje",
    "remekdjel",
    "rudjen",
    "rudjet",
    "sadje",
    "svagdje",
    "svidje",
    "svugdje",
    "sedmonedjelj",
    "sijedje",
    "sjedje",
    "smrdje",
    "snabdje",
    "snovidje",
    "starosjedje",
    "stidje",
    "studje",
    "sudjel",
    "tronedje",
    "ublijedje",
    "uvidje",
    "udjel",
    "udjen",
    "uprdje",
    "usidjel",
    "usjedje",
    "usmrdje",
    "uštedje",
    "cjelonedje",
    "četvoronedje",
    "čukundjed",
    "šestonedjelj",
    "štedje",
    "štogdje",
    "šukundjed",
];

pub const DIGRAPH_EXCEPTIONS_DZ: &[&str] = &[
    "feldžandarm",
    "nadžanj",
    "nadždrel",
    "nadžel",
    "nadžeo",
    "nadžet",
    "nadživ",
    "nadžinj",
    "nadžnj",
    "nadžrec",
    "nadžup",
    "odžali",
    "odžari",
    "odžel",
    "odžive",
    "odživljava",
    "odžubor",
    "odžvaka",
    "odžval",
    "odžvać",
    "podžanr",
    "podžel",
    "podže",
    "podžig",
    "podžiz",
    "podžil",
    "podžnje",
    "podžupan",
    "predželu",
    "predživot",
];

pub const DIGRAPH_EXCEPTIONS_NJ: &[&str] = &[
    "anjon",
    "injaric",
    "injekc",
    "injekt",
    "injicira",
    "injurij",
    "kenjon",
    "konjug",
    "konjunk",
    "nekonjug",
    "nekonjunk",
    "ssrnj",
    "tanjug",
    "vanjezičk",
];
