/*
 * Copyright 2021 Cargill Incorporated
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * -----------------------------------------------------------------------------
 */

//! Errors related to SQL-backed merkle-radix state representation.

use std::error::Error;
use std::fmt;

use crate::error::{InternalError, InvalidStateError};

/// This error may occur during the construction of a SqlMerkeState instance.
#[derive(Debug)]
pub enum SqlMerkleStateBuildError {
    InternalError(InternalError),
    InvalidStateError(InvalidStateError),
}

impl fmt::Display for SqlMerkleStateBuildError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            SqlMerkleStateBuildError::InternalError(e) => f.write_str(&e.to_string()),
            SqlMerkleStateBuildError::InvalidStateError(e) => f.write_str(&e.to_string()),
        }
    }
}

impl Error for SqlMerkleStateBuildError {
    fn source(&self) -> Option<&(dyn Error + 'static)> {
        match *self {
            SqlMerkleStateBuildError::InternalError(ref e) => Some(&*e),
            SqlMerkleStateBuildError::InvalidStateError(ref e) => Some(&*e),
        }
    }
}

impl From<InternalError> for SqlMerkleStateBuildError {
    fn from(err: InternalError) -> Self {
        SqlMerkleStateBuildError::InternalError(err)
    }
}

impl From<InvalidStateError> for SqlMerkleStateBuildError {
    fn from(err: InvalidStateError) -> Self {
        SqlMerkleStateBuildError::InvalidStateError(err)
    }
}
