use crate::{ChannelReceiver, ChannelSender};
use crate::message::SlackPayload;
use tokio::task::JoinHandle;

/// Provides a background worker task that sends the messages generated by the
/// layer.
pub(crate) async fn worker(mut rx: ChannelReceiver) {
    let client = reqwest::Client::new();
    while let Some(message) = rx.recv().await {
        match message {
            WorkerMessage::Data(payload) => {
                let webhook_url = payload.webhook_url().to_string();
                let payload =
                    serde_json::to_string(&payload).expect("failed to deserialize slack payload, this is a bug");
                match client.post(webhook_url).body(payload).send().await {
                    Ok(res) => {
                        tracing::debug!(?res);
                    }
                    Err(e) => {
                        tracing::error!(?e);
                    }
                };
            }
            WorkerMessage::Shutdown => {
                break;
            }
        }
    }
}

/// This worker manages a background async task that schedules the network requests to send traces
/// to the Slack on the running tokio runtime.
///
/// Ensure to invoke `.startup()` before, and `.teardown()` after, your application code runs. This
/// is required to ensure proper initialization and shutdown.
///
/// `tracing-layer-slack` synchronously generates payloads to send to the Slack API using the
/// tracing events from the global subscriber. However, all network requests are offloaded onto
/// an unbuffered channel and processed by a provided future acting as an asynchronous worker.
pub struct SlackBackgroundWorker {
    pub(crate) sender: ChannelSender,
    pub(crate) handle: JoinHandle<()>,
}

impl SlackBackgroundWorker {
    /// Initiate the worker's shutdown sequence.
    ///
    /// Without invoking`.teardown()`, your application may exit before all Slack messages can be
    /// sent.
    pub async fn shutdown(self) {
        self.sender.send(WorkerMessage::Shutdown).unwrap();
        self.handle.await.unwrap();
    }
}

#[derive(Debug)]
pub(crate) enum WorkerMessage {
    Data(SlackPayload),
    Shutdown,
}
