extern crate base64;
extern crate chrono;
extern crate chttp;
extern crate dialoguer;
extern crate humansize;
extern crate percent_encoding;
extern crate regex;
extern crate serde;
extern crate serde_json;

use dialoguer::theme::ColorfulTheme;
use dialoguer::Select;
use percent_encoding::{utf8_percent_encode, AsciiSet, CONTROLS};
use regex::{Captures, Regex};

use chrono::prelude::*;
use humansize::{file_size_opts as options, FileSize};
use serde::{Deserialize, Serialize};
use std::process::exit;
use structopt::StructOpt;
use yansi::Paint;

const FRAGMENT: &AsciiSet = &CONTROLS.add(b' ').add(b'"').add(b'<').add(b'>').add(b'`');

#[derive(StructOpt, Debug)]
#[structopt(name = "tpb - Search The Pirate Bay using the fastest available proxy")]
pub struct Options {
    /// Search for a torrent
    #[structopt(short, long)]
    pub query: Option<String>,

    #[structopt(short = "l", long = "show-proxy")]
    /// Shows list of proxies to use (first one is used when piping the output)
    pub show_proxy_list: bool,

    #[structopt(short = "m", long = "max-results")]
    /// Number of results to show (max 100)
    pub number: Option<i32>,
}

fn main() {
    let opt = Options::from_args();

    let _ = match opt {
        Options {
            query: Some(q),
            show_proxy_list: sl,
            number: n,
        } => handle_search(&q, sl, n),
        _ => {
            println!("Must pass a valid parameter");
            exit(1);
        }
    };
}

fn handle_search(query_unencoded: &str, show_proxy_list: bool, max_results: Option<i32>) {
    let _regex_parse_result_list = Regex::new(
        r#"(?m)href="/torrent/([^"]*)"[^\\>]*>([^<]*)<.*\n.*Uploaded ([^,]*), Size ([^,]*)"#,
    )
    .unwrap();

    let query = utf8_percent_encode(query_unencoded, FRAGMENT).to_string();

    status_s(format!("Search for '{}'...", query_unencoded));

    // render proxy to choose from
    let proxy_domain;

    proxy_domain = match prompt_or_pick_proxy(show_proxy_list) {
        Some(value) => value,
        None => return,
    };

    status_s(format!("Using proxy {}...", proxy_domain));

    let proxy_search = format!(
        "https://{}/newapi/q.php?q={}&cat=",
        proxy_domain,
        query.as_str()
    );

    status_s(format!("GET {}", proxy_search));

    let result_json: String = chttp::get(proxy_search).unwrap().body_mut().text().unwrap();
    let result_items: Vec<PirateBayApiResponseElement> = match serde_json::from_str(&result_json) {
        Ok(e) => e,
        Err(_) => {
            if !show_proxy_list {
                return handle_search(query_unencoded, show_proxy_list, max_results);
            }
            return;
        }
    };

    let mut count = 0;
    let is_output_redirected = !atty::is(atty::Stream::Stdout);

    for mats in result_items.into_iter() {
        let mat: PirateBayApiResponseElement = mats;

        if mat.id == "0" {
            continue;
        }

        let magnet_link = format!(
            "magnet:?xt=urn:btih:{}&dn={}&tr=udp%3A%2F%2Ftracker.coppersurfer.tk%3A6969%2Fannounce&tr=udp%3A%2F%2F9.rarbg.me%3A2850%2Fannounce&tr=udp%3A%2F%2F9.rarbg.to%3A2920%2Fannounce&tr=udp%3A%2F%2Ftracker.opentrackr.org%3A1337&tr=udp%3A%2F%2Ftracker.leechers-paradise.org%3A6969%2Fannounce",
            mat.info_hash, utf8_percent_encode(&mat.name, FRAGMENT).to_string());
        let uploaded = get_date(&mat.added);
        let size_i64: i64 = mat.size.parse().unwrap_or(0);
        let size = size_i64.file_size(options::CONVENTIONAL).unwrap();
        let title = mat.name;

        if is_output_redirected {
            println!("[{} {}] {} {}", size, uploaded, title, magnet_link);
        } else {
            eprintln!(
                "{} {} ({}) > {}",
                Paint::yellow(size).dimmed(),
                Paint::yellow(title).bold(),
                Paint::yellow(uploaded).italic(),
                Paint::cyan(magnet_link).dimmed()
            );
        }

        count += 1;
        if let Some(max_results_to_show) = max_results {
            if max_results_to_show > 0 && count >= max_results_to_show {
                break;
            }
        }
    }

    status_s(format!("{} result found", count));
    status("Use the magnet link to download the torrent.");
}

fn prompt_or_pick_proxy(show_proxy_list: bool) -> Option<String> {
    let mut options: Vec<&str> = Vec::new();
    let regex_proxy_list = Regex::new(r"(?m)reportLink\('([^']*)'\)").unwrap();

    // fetch proxy list
    status("Finding proxy ...");

    let proxy_html = chttp::get("https://proxybay.github.io/")
        .unwrap()
        .body_mut()
        .text()
        .unwrap();

    let proxy_result = regex_proxy_list.captures_iter(proxy_html.as_str());
    let captures: Vec<Captures> = proxy_result.collect();
    for cap in &captures {
        let item = cap.get(1).unwrap().as_str();
        if item.len() > 0 {
            options.push(item);
        }
    }

    let is_output_redirected = !atty::is(atty::Stream::Stdout);
    if is_output_redirected || !show_proxy_list {
        return Some(
            captures
                .first()
                .expect("Could not parse list of proxies...")[1]
                .into(),
        );
    } else {
        let selected_proxy = Select::with_theme(&ColorfulTheme::default())
            .with_prompt("Select the proxy")
            .default(0)
            .items(&options)
            .interact_opt()
            .unwrap();
        if let Some(selected_proxy) = selected_proxy {
            return Some(captures.get(selected_proxy).unwrap()[1].into());
        } else {
            status("You did not select a proxy ...");
            return None;
        }
    }
}

fn get_date(d: &str) -> String {
    // Convert the timestamp string into an i64
    let timestamp: i64 = d.parse().unwrap();

    // Create a NaiveDateTime from the timestamp
    let naive = NaiveDateTime::from_timestamp(timestamp, 0);

    // Create a normal DateTime from the NaiveDateTime
    let datetime: DateTime<Utc> = DateTime::from_utc(naive, Utc);

    // Format the datetime how you want
    return datetime.format("%Y-%m-%d %H:%M:%S").to_string();
}

fn status(text: &str) {
    status_s(String::from(text));
}

fn status_s(text: String) {
    eprintln!("{}", Paint::white(text).dimmed());
}

#[derive(Debug, Serialize, Deserialize)]
pub struct PirateBayApiResponseElement {
    id: String,
    name: String,
    info_hash: String,
    size: String,
    added: String,
}
