// Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT OR Apache-2.0

use std::fs::File;
use tempfile::TempDir;
use test_utils::{dir_url, read_to_end, test_data};
use tough::{FilesystemTransport, Limits, Repository, RepositoryLoader};

mod test_utils;

/// Test that `tough` can process repositories generated by [`tuf`], the reference Python
/// implementation using the `load_default` function.
///
/// [`tuf`]: https://github.com/theupdateframework/tuf
#[test]
fn test_tuf_reference_impl() {
    let base = test_data().join("tuf-reference-impl");

    let repo = RepositoryLoader::new(
        File::open(base.join("metadata").join("1.root.json")).unwrap(),
        dir_url(base.join("metadata")),
        dir_url(base.join("targets")),
    )
    .load()
    .unwrap();
    assert_tuf_reference_impl(&repo);
}

fn assert_tuf_reference_impl(repo: &Repository) {
    assert_eq!(
        read_to_end(repo.read_target("file1.txt").unwrap().unwrap()),
        &b"This is an example target file."[..]
    );
    assert_eq!(
        read_to_end(repo.read_target("file2.txt").unwrap().unwrap()),
        &b"This is an another example target file."[..]
    );
    assert_eq!(
        repo.targets()
            .signed
            .targets
            .get("file1.txt")
            .unwrap()
            .custom
            .get("file_permissions")
            .unwrap(),
        "0644"
    );

    assert!(repo
        .targets()
        .signed
        .delegations
        .as_ref()
        .unwrap()
        .target_is_delegated(&"file3.txt".to_string()));
}

/// Test that `tough` can process repositories generated by [`tuf`], the reference Python
/// implementation using the `load` function with non-default [`Options`].
#[test]
fn test_tuf_reference_impl_default_transport() {
    let base = test_data().join("tuf-reference-impl");
    let datastore = TempDir::new().unwrap();

    let repo = RepositoryLoader::new(
        File::open(base.join("metadata").join("1.root.json")).unwrap(),
        dir_url(base.join("metadata")),
        dir_url(base.join("targets")),
    )
    .transport(FilesystemTransport)
    .limits(Limits {
        max_root_size: 1000,
        max_targets_size: 2000,
        max_timestamp_size: 3000,
        max_root_updates: 1,
    })
    .datastore(datastore.path())
    .load()
    .unwrap();
    assert_tuf_reference_impl(&repo);
}
