extern crate rss;

use core::str::FromStr;
use std::collections::BTreeMap;
use std::convert::TryFrom;

use chrono::Duration;
use rss::extension::Extension;
use rss::Item;
#[cfg(any(feature = "parse-names", feature = "require-parse-names"))]
use torrent_name_parser::Metadata;

use crate::Error;

#[derive(Clone, Debug, Default, PartialEq)]
pub struct Torrent {
	pub name: String,
	#[cfg(all(feature = "parse-names", not(feature = "require-parse-names")))]
	pub metadata: Option<Metadata>,
	#[cfg(feature = "require-parse-names")]
	pub metadata: Metadata,
	pub size: u64,
	pub categories: Vec<u32>, // In theory, u32 is good for our use case
	pub link: String,
	pub seeders: Option<u16>,
	pub leechers: Option<u16>,
	pub minimum_ratio: Option<f32>,
	pub minimum_seedtime: Option<Duration>
}

fn get_extension_value<'a>(ext: &'a BTreeMap<String, Vec<Extension>>, key: &str) -> Result<Option<&'a str>, Error> {
	let ext = match ext.get("attr") {
		Some(v) => v,
		None => return Ok(None)
	};
	if(ext.is_empty()) {
		return Err(Error::EmptyExtension(key.to_string()))
	}
	for extension in ext.iter() {
		if let Some(name) = extension.attrs().get("name") {
			if let Some(value) = extension.attrs().get("value") {
				if(name == key) {
					return Ok(Some(value));
				}
			}
		}
	}
	Ok(None)
}

fn get_parsed_extension_value<T>(ext: &BTreeMap<String, Vec<Extension>>, key: &str) -> Result<Option<T>, Error>
where
	T: FromStr,
	Error: From<T::Err>
{
	match get_extension_value(ext, key)? {
		Some(v) => Ok(Some(v.parse::<T>()?)),
		None => Ok(None)
	}
}

impl TryFrom<Item> for Torrent {
	type Error = Error;
	fn try_from(item: Item) -> Result<Self, Self::Error> {
		let mut this = Self{
			name: item.title().ok_or(Error::MissingTitle)?.to_string(),
			size: item.enclosure().ok_or(Error::MissingSize)?.length().parse()?,
			categories: item.categories().iter().map(|category| category.name().parse::<u32>()).collect::<Result<Vec<_>, _>>()?,
			link: item.link().ok_or(Error::MissingLink)?.to_string(),
			..Default::default()
		};
		#[cfg(all(feature = "parse-names", not(feature = "require-parse-names")))]
		{
			this.metadata = Metadata::from(&this.name).ok();
		}
		#[cfg(feature = "require-parse-names")]
		{
			this.metadata = match Metadata::from(&this.name) {
				Ok(v) => v,
				Err(_) => return Err(Self::Error::parse_name_failure(&this.name))
			};
		}
		if let Some(torznab) = item.extensions().get("torznab") {
			this.seeders = match get_parsed_extension_value(torznab, "seeders")? {
				Some(seeders) => {
					this.leechers = get_parsed_extension_value::<u16>(torznab, "peers")?.map(|peers| peers - seeders);
					Some(seeders)
				},
				None => None
			};
			this.minimum_ratio = get_parsed_extension_value(torznab, "minimumrato")?;
			this.minimum_seedtime = get_parsed_extension_value(torznab, "minimumseedtime")?.map(Duration::seconds);
		}
		Ok(this)
	}
}

