//! Errors from [SqlBuilder](crate::sql_builder::SqlBuilder)
use std::fmt;

#[derive(Debug)]
/// Represents all errors from the SQL Builder
pub enum SqlBuilderError {
    /// The field is not mapped to a column or SQL expression. Contains the field name.
    FieldMissing(String),
    /// The field is not mapped to a column or SQL expression. Contains the field name.
    PredicateMissing(String),
    /// The join is not mapped to a column or SQL expression. Contains the join name and the table name.
    JoinMissing(String, String),
    /// The merge is not mapped. Contains the field name.
    MergeMissing(String),
    /// The selection is not known to the mapper. Contains the field name.
    SelectionMissing(String),
    /// The field requires a role that the query does not have. Contains the role and the query_path.
    RoleRequired(String, String),
    /// The filter expects other arguments. Typically raised by custom functions (FN) if the number of arguments is wrong.
    FilterInvalid(String),
    /// A query expression requires a query parameter, that is not provided. Contains the parameter.
    QueryParamMissing(String),
    /// The query parameter that is required by the query expression is wrong. Contains the parameter and the details.
    QueryParamInvalid(String, String),
    /// A predicate requires more arguments, than the toql q uery provided, contains the predicate.
    PredicateArgumentMissing(String),

    /// An key cannot be set, because type is wrong or key is composite key
    KeyMismatch(String, String),

    /// A path was found for a selection that only exists in root, such as $all, $mut, $cnt
    PathUnexpected(String),
}

impl fmt::Display for SqlBuilderError {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            SqlBuilderError::FieldMissing(ref s) => write!(f, "field `{}` is missing", s),
            SqlBuilderError::PathUnexpected(ref s) => {
                write!(f, "a path `{}` was found but no path is allowed", s)
            }
            SqlBuilderError::SelectionMissing(ref s) => write!(f, "selection `{}` is missing", s),
            SqlBuilderError::PredicateMissing(ref s) => write!(f, "predicate `@{}` is missing", s),
            SqlBuilderError::JoinMissing(ref s, ref t) => {
                write!(f, "join `{}` is missing on mapper for table `{}`", s, t)
            }
            SqlBuilderError::MergeMissing(ref s) => write!(f, "merge `{}` is missing", s),
            SqlBuilderError::RoleRequired(ref s, ref p) => {
                write!(f, "role `{}` is required for `{}`", s, p)
            }
            SqlBuilderError::FilterInvalid(ref s) => write!(f, "filter `{}` is invalid ", s),
            SqlBuilderError::KeyMismatch(ref t, ref s) => {
                write!(f, "Key with value `{}` does not match key of `{}` ", t, s)
            }
            SqlBuilderError::QueryParamMissing(ref s) => {
                write!(f, "query parameter `{}` is missing ", s)
            }
            SqlBuilderError::QueryParamInvalid(ref s, ref d) => {
                write!(f, "query parameter `{}` is invalid: {} ", s, d)
            }
            SqlBuilderError::PredicateArgumentMissing(ref s) => {
                write!(f, "predicate `{}` requires more arguments. ", s)
            }
        }
    }
}
