//! An argument for SQL expressions.

pub mod error;
pub mod from;
pub mod from_row;
pub mod try_into;

/// Enum to keep the different argument types.
#[derive(Clone, Debug, PartialEq)]
pub enum SqlArg {
    U64(u64),
    I64(i64),
    F64(f64),
    Str(String),
    Bool(bool),
    Null,
}

impl SqlArg {
    /// Build SQL string.
    pub fn to_sql_string(&self) -> String {
        match self {
            SqlArg::U64(t) => t.to_string(),
            SqlArg::I64(t) => t.to_string(),
            SqlArg::F64(t) => t.to_string(),
            SqlArg::Str(t) => format!("'{}'", t.replace("'", "''")),
            SqlArg::Bool(t) => t.to_string(),
            SqlArg::Null => "NULL".to_string(),
        }
    }

    /// Return i64 or None, if type mismatches.
    pub fn get_i64(&self) -> Option<i64> {
        if let Self::I64(v) = self {
            Some(v.to_owned())
        } else {
            None
        }
    }
    /// Return f64 or None, if type mismatches.
    pub fn get_f64(&self) -> Option<f64> {
        if let Self::F64(v) = self {
            Some(v.to_owned())
        } else {
            None
        }
    }
    /// Return bool or None, if type mismatches.
    pub fn get_bool(&self) -> Option<bool> {
        if let Self::Bool(v) = self {
            Some(v.to_owned())
        } else {
            None
        }
    }
    /// Return u64 or None, if type mismatches.
    pub fn get_u64(&self) -> Option<u64> {
        if let Self::U64(v) = self {
            Some(v.to_owned())
        } else {
            None
        }
    }
    /// Return str or None, if type mismatches.
    pub fn get_str(&self) -> Option<&str> {
        if let Self::Str(v) = self {
            Some(v)
        } else {
            None
        }
    }

    /// Returns true, if argument is null.
    pub fn is_null(&self) -> bool {
        matches!(self, Self::Null)
    }
    /// Returns true, if argument is string and matches other string.
    pub fn cmp_str(&self, other: &str) -> bool {
        if let Self::Str(v) = self {
            v == other
        } else {
            false
        }
    }
}

impl ToString for SqlArg {
    fn to_string(&self) -> String {
        match self {
            SqlArg::U64(t) => t.to_string(),
            SqlArg::I64(t) => t.to_string(),
            SqlArg::F64(t) => t.to_string(),
            SqlArg::Str(t) => t.to_string(),
            SqlArg::Bool(t) => t.to_string(),
            SqlArg::Null => "NULL".to_string(),
        }
    }
}

/// Returns true, if list of arguments would be a valid key.
pub fn valid_key(args: &[SqlArg]) -> bool {
    let contains_zero_key = args.iter().any(|a| match a {
        SqlArg::U64(x) => x == &0,
        SqlArg::Str(x) => x.is_empty(),
        SqlArg::Null => true,
        _ => false,
    });
    !contains_zero_key
}
