/*
Copyright (C) 2020-2021 Kunal Mehta <legoktm@debian.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

use ini::Ini;
use std::{fmt, fs};

/// Fields needed to connect to the MariaDB databases on Toolforge
///
/// Calling `.to_string()` will get the connection info in the format of the
/// [`mysql`](https://docs.rs/mysql) crate.
pub struct DBConnectionInfo {
    pub database: String,
    pub host: String,
    pub user: String,
    pub password: String,
}

struct ReplicaMyCnf {
    user: String,
    password: String,
}

impl fmt::Display for DBConnectionInfo {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        let url = format!(
            "mysql://{}:{}@{}:3306/{}",
            self.user, self.password, self.host, self.database
        );
        write!(f, "{}", url)
    }
}

/// Database clusters to connect to
///
/// See [Connecting to the database replicas](https://wikitech.wikimedia.org/wiki/Help:Toolforge/Database#Connecting_to_the_database_replicas)
/// for information about the differences between the web and analytics
/// clusters.
#[derive(PartialEq, Eq)]
pub enum ToolsDBCluster {
    ANALYTICS,
    WEB,
}

/// Get database connection info. Note that the `connection_info!()` macro
/// is a simpler way to call this.
pub fn get_db_connection_info(
    dbname: &str,
    cluster: ToolsDBCluster,
) -> crate::Result<DBConnectionInfo> {
    let domain = format!(
        "{}.db.svc.wikimedia.cloud",
        match cluster {
            ToolsDBCluster::ANALYTICS => "analytics",
            ToolsDBCluster::WEB => "web",
        }
    );
    let normalized_dbname = dbname.trim_end_matches("_p");
    let host = if normalized_dbname == "meta" {
        format!("s7.{}", domain)
    } else {
        format!("{}.{}", normalized_dbname, domain)
    };

    let my_cnf = load_replica_cnf()?;

    Ok(DBConnectionInfo {
        database: format!("{}_p", normalized_dbname),
        host,
        user: my_cnf.user,
        password: my_cnf.password,
    })
}

fn load_replica_cnf() -> crate::Result<ReplicaMyCnf> {
    #[cfg(not(test))]
    let path = dirs::home_dir()
        // Just panic if this happens, seriously
        .expect("Couldn't find home directory")
        .join("replica.my.cnf");
    // Use a different file in test contexts
    #[cfg(test)]
    let path = std::path::PathBuf::new().join("./tests/replica.my.cnf");
    if !path.exists() {
        return Err(crate::Error::NotToolforge("replica.my.cnf".to_string()));
    }
    let contents = fs::read_to_string(path)?;
    let conf = Ini::load_from_str(&contents)?;
    // TODO: avoid unwrap
    let section = conf.section(Some("client")).unwrap();
    Ok(ReplicaMyCnf {
        user: section.get("user").unwrap().to_string(),
        password: section.get("password").unwrap().to_string(),
    })
}

#[cfg(test)]
mod tests {

    #[test]
    fn test_to_string() {
        let info = super::DBConnectionInfo {
            database: "meta_p".to_string(),
            host: "hostname".to_string(),
            user: "u12345".to_string(),
            password: "correcthorsebatterystaple".to_string(),
        };
        assert_eq!(
            "mysql://u12345:correcthorsebatterystaple@hostname:3306/meta_p"
                .to_string(),
            info.to_string()
        )
    }

    #[test]
    fn test_connection_info() -> crate::Result<()> {
        use crate::connection_info;

        assert_eq!(
            "mysql://u12345:correcthorsebatterystaple@enwiki.web.db.svc.wikimedia.cloud:3306/enwiki_p".to_string(),
            connection_info!("enwiki")?.to_string()
        );
        assert_eq!(
            "mysql://u12345:correcthorsebatterystaple@enwiki.analytics.db.svc.wikimedia.cloud:3306/enwiki_p"
                .to_string(),
            connection_info!("enwiki_p", ANALYTICS)?.to_string()
        );
        assert_eq!(
            "mysql://u12345:correcthorsebatterystaple@s7.web.db.svc.wikimedia.cloud:3306/meta_p".to_string(),
            connection_info!("meta_p")?.to_string()
        );

        Ok(())
    }
}
