/*
Small library for common tasks on Wikimedia Toolforge
Copyright (C) 2013, 2017, 2020 Kunal Mehta <legoktm@member.fsf.org>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
//! The `toolforge` crate provides helper functions for various tasks on
//! [Toolforge](https://toolforge.org/).
//!
//! See [wikitech](https://wikitech.wikimedia.org/wiki/User:Legoktm/toolforge_library)
//! for the full documentation.

#[cfg(feature = "mysql")]
pub mod db;
#[cfg(feature = "mysql")]
mod error;

#[cfg(feature = "mysql")]
pub use error::Error;
#[cfg(feature = "mysql")]
pub type Result<A> = std::result::Result<A, crate::Error>;

/// Macro to generate a Tool-specific `User-Agent` header in compliance with
/// [Wikimedia's User-Agent policy](https://meta.wikimedia.org/wiki/User-Agent_policy).
/// The first and only required parameter is the tool's name. Optional second
/// and third parameters can be overrides to the url and email, respectively.
///
/// # Example
/// ```rust
/// # use toolforge::user_agent;
/// assert_eq!(
///     user_agent!("mycooltool"),
///     "https://mycooltool.toolforge.org/ tools.mycooltool@tools.wmflabs.org"
/// );
/// ```
///
///
#[macro_export]
macro_rules! user_agent {
    ( $tool:expr ) => {
        concat!(
            "https://",
            $tool,
            ".toolforge.org/ tools.",
            $tool,
            "@tools.wmflabs.org"
        )
    };
    ( $tool:expr, $url:expr ) => {
        concat!($url, " tools.", $tool, "@tools.wmflabs.org")
    };
    ( $tool:expr, $url:expr, $email:expr ) => {
        concat!($url, " ", $email)
    };
}

/// Get connection information for the [Toolforge databases](https://wikitech.wikimedia.org/wiki/Help:Toolforge/Database).
///
/// The first argument is the database name, with or without a `_p` suffix.
/// The second (optional) argument is which cluster to connect to, either
/// `WEB` (default) or `ANALYTICS`.
///
/// You can also connect to the the [`meta_p`](https://wikitech.wikimedia.org/wiki/Help:Toolforge/Database#Metadata_database)
/// database this way.
///
/// ```rust
/// # fn demo() -> toolforge::Result<()> {
/// assert_eq!(
///     "mysql://user:pass@enwiki.web.db.svc.eqiad.wmflabs:3306/enwiki_p".to_string(),
///     toolforge::connection_info!("enwiki")?.to_string()
/// );
/// assert_eq!(
///     "mysql://user:pass@enwiki.analytics.db.svc.eqiad.wmflabs:3306/enwiki_p".to_string(),
///     toolforge::connection_info!(
///         "enwiki",
///         ANALYTICS
///     )?.to_string()
/// );
/// # Ok(())
/// # }
/// ```
/// Note: This requires the `mysql` feature to be enabled.
#[cfg(feature = "mysql")]
#[macro_export]
macro_rules! connection_info {
    ( $dbname:expr ) => {
        $crate::db::get_db_connection_info(
            $dbname,
            $crate::db::ToolsDBCluster::WEB,
        )
    };
    ( $dbname:expr, $cluster:ident ) => {
        $crate::db::get_db_connection_info(
            $dbname,
            $crate::db::ToolsDBCluster::$cluster,
        )
    };
}

#[cfg(test)]
mod tests {
    use super::user_agent;

    const USER_AGENT: &str = user_agent!("foo");

    #[test]
    fn test_const_user_agent() {
        assert_eq!(
            USER_AGENT,
            "https://foo.toolforge.org/ tools.foo@tools.wmflabs.org"
        );
    }

    #[test]
    fn test_user_agent() {
        assert_eq!(
            user_agent!("foo"),
            "https://foo.toolforge.org/ tools.foo@tools.wmflabs.org"
        );
        assert_eq!(
            user_agent!("foo", "https://example.org/"),
            "https://example.org/ tools.foo@tools.wmflabs.org"
        );
        assert_eq!(
            user_agent!("foo", "https://example.org/", "tool@example.org"),
            "https://example.org/ tool@example.org"
        );
    }
}
