use std::env;
use std::net::IpAddr;
use std::process;
use std::str::FromStr;
use toluol::DnsType;

#[derive(Debug, PartialEq)]
pub enum ConnectionType {
    Udp,
    Tcp,
    #[cfg(feature = "tls")]
    Tls,
    #[cfg(feature = "http")]
    HttpGet,
    #[cfg(feature = "http")]
    HttpPost,
    #[cfg(feature = "http")]
    HttpsGet,
    #[cfg(feature = "http")]
    HttpsPost,
}

#[derive(Debug)]
pub struct Args {
    pub nameserver: String,
    pub url: String,
    pub qtype: DnsType,
    pub verbose: bool,
    #[cfg(feature = "json")]
    pub json: bool,
    pub print_meta: bool,
    pub pad_answers: bool,
    pub use_dnssec: bool,
    pub connection_type: ConnectionType,
    pub port: u16,
}

enum ConsumeNext {
    Port,
}

const DEFAULT_NAMESERVER: &str = "ordns.he.net";
const DEFAULT_URL: &str = "example.com";
const DEFAULT_QTYPE: DnsType = DnsType::AAAA;

pub fn parse_args() -> Args {
    // skip executable name
    let args: Vec<String> = env::args().skip(1).collect();

    let mut nameserver = DEFAULT_NAMESERVER.into();
    let mut url = DEFAULT_URL.into();
    let mut qtype = DEFAULT_QTYPE;
    let mut verbose = false;
    #[cfg(feature = "json")]
    let mut json = false;
    let mut print_meta = true;
    let mut pad_answers = true;
    let mut use_dnssec = false;
    let mut connection_type = ConnectionType::Udp;
    let mut port = None;

    let mut reverse = false;
    let mut consume_next = None;

    for arg in args {
        if let Some(to_consume) = &consume_next {
            match to_consume {
                ConsumeNext::Port => match arg.parse::<u16>() {
                    Ok(val) => port = Some(val),
                    Err(_) => err(format!("Invalid port: {}.", arg)),
                },
            }
            consume_next = None;
        } else if let Some(ns) = arg.strip_prefix('@') {
            // nameserver
            nameserver = ns.to_string();
        } else if let Some(flag) = arg.strip_prefix('+') {
            // flags
            match flag {
                "verbose" => {
                    verbose = true;
                }
                #[cfg(feature = "json")]
                "json" => {
                    json = true;
                }
                "no-meta" => {
                    print_meta = false;
                }
                "no-padding" => {
                    pad_answers = false;
                }
                "do" => {
                    use_dnssec = true;
                }
                "tcp" => {
                    connection_type = ConnectionType::Tcp;
                }
                #[cfg(feature = "tls")]
                "dot" | "tls" => {
                    connection_type = ConnectionType::Tls;
                }
                #[cfg(feature = "http")]
                "http-get" => {
                    connection_type = ConnectionType::HttpGet;
                }
                #[cfg(feature = "http")]
                "http" | "http-post" => {
                    connection_type = ConnectionType::HttpPost;
                }
                #[cfg(feature = "http")]
                "https-get" => {
                    connection_type = ConnectionType::HttpsGet;
                }
                #[cfg(feature = "http")]
                "doh" | "https" | "https-post" => {
                    connection_type = ConnectionType::HttpsPost;
                }
                x => {
                    err(format!("Invalid flag: +{}.", x));
                }
            }
        } else if let Some(option) = arg.strip_prefix('-') {
            // options
            match option {
                "h" | "-help" => {
                    print_help();
                    process::exit(0);
                }
                "p" | "-port" => {
                    consume_next = Some(ConsumeNext::Port);
                }
                "V" | "-version" => {
                    print_version();
                    process::exit(0);
                }
                "x" => {
                    reverse = true;
                }
                x => {
                    err(format!("Invalid option: -{}.", x));
                }
            }
        } else {
            match DnsType::from_str(&arg.to_uppercase()) {
                Ok(t) => {
                    qtype = t;
                }
                Err(_) => {
                    // use URL as fallback
                    url = arg;
                }
            }
        }
    }

    if verbose && !pad_answers {
        err("Cannot use both +verbose and +no-padding.");
    }

    if reverse {
        match IpAddr::from_str(url.as_str()) {
            Err(_) => {
                err(format!(
                    "Expected IP address for reverse lookup, but got: {}.",
                    url
                ));
            }
            Ok(IpAddr::V4(addr)) => {
                let octets = addr.octets();
                url = format!(
                    "{}.{}.{}.{}.in-addr.arpa",
                    octets[3], octets[2], octets[1], octets[0]
                );
            }
            Ok(IpAddr::V6(addr)) => {
                url = String::with_capacity(72);
                for s in addr.segments().iter().rev() {
                    for c in format!("{:04x}", s).chars().rev() {
                        url.push(c);
                        url.push('.');
                    }
                }
                url.push_str("ip6.arpa");
            }
        }
        qtype = DnsType::PTR;
    }

    #[cfg(feature = "tls")]
    if connection_type == ConnectionType::Tls {
        if webpki::DnsNameRef::try_from_ascii_str(&nameserver).is_err() {
            err("Nameserver must be a valid hostname for TLS.");
        }
        if port.is_none() {
            port = Some(853);
        }
    }

    #[cfg(feature = "http")]
    if port.is_none() {
        if [ConnectionType::HttpGet, ConnectionType::HttpPost].contains(&connection_type) {
            port = Some(80);
        } else if [ConnectionType::HttpsGet, ConnectionType::HttpsPost].contains(&connection_type) {
            port = Some(443);
        }
    }

    Args {
        nameserver,
        url,
        qtype,
        verbose,
        #[cfg(feature = "json")]
        json,
        print_meta,
        pad_answers,
        use_dnssec,
        connection_type,
        port: port.unwrap_or(53),
    }
}

fn print_help() {
    println!("Usage: \ttoluol [@nameserver] [domain] [q-type] [options] [flags]");
    println!();
    println!("Where: \tq-type is a valid DNS QTYPE (AAAA, A, TXT, MX, SOA, ...)");
    println!();
    println!("       \toptions is one or more of the following:");
    println!("       \t    -h | --help            (print this help message)");
    println!("       \t    -V | --version         (print the version of toluol)");
    println!("       \t    -p | --port <port>     (use the given port number)");
    println!("       \t    -x                     (shortcut for reverse lookup)");
    println!();
    println!("       \tflags is one or more of the following:");
    println!("       \t    +verbose     (print all sections, i.e. header, OPT, and question)");
    #[cfg(feature = "json")]
    println!("       \t    +json        (format output as JSON; may be used with +verbose)");
    println!("       \t    +no-meta     (don't print query metadata, e.g. server and time)");
    println!("       \t    +no-padding  (don't pad output; cannot be used with +verbose)");
    println!("       \t    +do          (fetch DNSSEC records)");
    println!("       \t    +tcp         (use TCP instead of UDP)");
    #[cfg(feature = "tls")]
    {
        println!("       \t    +dot         (use DNS over TLS)");
        println!("       \t    +tls         (use DNS over TLS)");
    }
    #[cfg(feature = "http")]
    {
        println!("       \t    +doh         (use DNS over HTTPS, with POST)");
        println!("       \t    +https       (use DNS over HTTPS, with POST)");
        println!("       \t    +https-post  (use DNS over HTTPS, with POST)");
        println!("       \t    +https-get   (use DNS over HTTPS, with GET)");
        println!("       \t    +http        (use DNS over HTTP, with POST)");
        println!("       \t    +http-post   (use DNS over HTTP, with POST)");
        println!("       \t    +http-get    (use DNS over HTTP, with GET)");
    }
    println!();
    println!("Note: order of the arguments does not matter.");
    println!();
    println!(
        "If no arguments are specified, the default behaviour is `toluol @{} {} {}`.",
        DEFAULT_NAMESERVER, DEFAULT_URL, DEFAULT_QTYPE
    );
}

fn print_version() {
    println!("toluol v{}", env!("CARGO_PKG_VERSION"));
}

fn err(msg: impl AsRef<str>) {
    eprintln!("{}", msg.as_ref());
    process::exit(1);
}
