#! /bin/sh

CRATE_NAME=toluol

BOLD_RED="\033[1;31m"
GREEN="\033[0;32m"
YELLOW="\033[0;93m"
RESET_COLOURS="\033[0m"
DONE="${GREEN} done.${RESET_COLOURS}"
ABORT="${BOLD_RED}Aborting.${RESET_COLOURS}"

read_and_check_input() {
    echo -n "$1 (y/N) "
    read input
    if [[ "$input" != "y" && "$input" != "Y" ]]; then
        abort
    fi
}

abort() {
    echo -e $ABORT
    exit 1
}

echo -n "Regenerating Cargo.lock to make sure it is up to date..."
cargo generate-lockfile --quiet
echo -e $DONE

if [[ -n "$(git status --porcelain)" ]]; then
    if [[ -z "$(git status --untracked-files=no --porcelain)" ]]; then 
        echo -e "${YELLOW}There are no uncommited changes, but there are untracked files.${RESET_COLOURS}"
        read_and_check_input "Do you still want to continue?"
    else 
        echo -e "${BOLD_RED}There are uncommitted changes.${RESET_COLOURS}"
        abort
    fi
else 
    echo -e "${GREEN}Working directory clean.${RESET_COLOURS}"
fi

version=$(sed 's/^version = "\(.*\)"$/\1/;t;d' Cargo.toml)
read_and_check_input "Detected version $version. Is that correct?"

echo -n 'Dry-running `cargo publish`...'
if cargo publish --dry-run --allow-dirty --quiet ; then
    echo -e " ${GREEN}successful.${RESET_COLOURS}"
else
    echo -e " ${BOLD_RED}failed!${RESET_COLOURS}"
    abort
fi

echo -n "Tagging HEAD as $version... "
git tag "v$version"
echo -e $DONE

echo -n "Pushing to origin..."
git push --quiet
echo -e $DONE

echo -n "Publishing new version on crates.io..."
cargo publish
echo -e $DONE

# no idea if this is necessary, but can't hurt
echo -n "Waiting for crates.io to update..."
sleep 5
echo -e $DONE

echo -n "Downloading new .crate file..."
curl "https://crates.io/api/v1/crates/$CRATE_NAME/$version/download" -s -L -o __crate
echo -e $DONE

if ! [[ "$(file crate)" =~ ^.*gzip.*$ ]]; then
    echo -e "${BOLD_RED}The downloaded file is not a gzip archive!${RESET_COLOURS}"
    abort
fi

# remove filename at the end of output and only grab the actual checksum
sha=$(sha512sum __crate | sed 's/^\(.*\) .*$/\1/')
echo "The sha512sum of the new .crate file is $sha."

echo -e "${YELLOW}Now go update the AUR package.${RESET_COLOURS}"
read tmp

echo -n "Removing downloaded .crate file..."
rm __crate
echo -e $DONE
