use std::env;
use std::net::IpAddr;
use std::process;
use std::str::FromStr;
use toluol::DnsType;

#[derive(Debug, PartialEq)]
pub enum ConnectionType {
    UDP,
    TCP,
    #[cfg(feature = "tls")]
    TLS,
}

#[derive(Debug)]
pub struct Args {
    pub nameserver: String,
    pub url: String,
    pub qtype: DnsType,
    pub short: bool,
    pub use_dnssec: bool,
    pub connection_type: ConnectionType,
    pub port: u16,
}

enum ConsumeNext {
    Port,
}

pub fn parse_args() -> Args {
    // skip executable name
    let args: Vec<String> = env::args().skip(1).collect();

    let mut nameserver = "1.1.1.1".to_string();
    let mut url = "example.com".to_string();
    let mut qtype = DnsType::A;
    let mut short = false;
    let mut use_dnssec = false;
    let mut connection_type = ConnectionType::UDP;
    let mut port = None;

    let mut reverse = false;
    let mut consume_next = None;

    for arg in args {
        if let Some(to_consume) = &consume_next {
            match to_consume {
                ConsumeNext::Port => match arg.parse::<u16>() {
                    Ok(val) => port = Some(val),
                    Err(_) => err("Invalid port."),
                },
            }
            consume_next = None;
        } else if let Some(ns) = arg.strip_prefix('@') {
            // nameserver
            nameserver = ns.to_string();
        } else if let Some(flag) = arg.strip_prefix('+') {
            // flags
            match flag {
                "short" => {
                    short = true;
                }
                "do" => {
                    use_dnssec = true;
                }
                "tcp" => {
                    connection_type = ConnectionType::TCP;
                }
                #[cfg(feature = "tls")]
                "tls" => {
                    connection_type = ConnectionType::TLS;
                }
                x => {
                    err(format!("Invalid flag: {}.", x));
                }
            }
        } else if let Some(option) = arg.strip_prefix('-') {
            // options
            match option {
                "h" | "-help" => {
                    print_help();
                    process::exit(0);
                }
                "p" | "-port" => {
                    consume_next = Some(ConsumeNext::Port);
                }
                "x" => {
                    reverse = true;
                }
                x => {
                    eprintln!("Invalid option: {}.", x);
                    process::exit(1);
                }
            }
        } else {
            match DnsType::from_str(&arg.to_uppercase()) {
                Ok(t) => {
                    qtype = t;
                }
                Err(_) => {
                    // use URL as fallback
                    url = arg;
                }
            }
        }
    }

    if reverse {
        match IpAddr::from_str(url.as_str()) {
            Err(_) => {
                eprintln!("Invalid IP address: {}.", url);
                process::exit(1);
            }
            Ok(IpAddr::V4(addr)) => {
                let octets = addr.octets();
                url = format!(
                    "{}.{}.{}.{}.in-addr.arpa",
                    octets[3], octets[2], octets[1], octets[0]
                );
                qtype = DnsType::PTR;
            }
            Ok(IpAddr::V6(addr)) => {
                url = String::with_capacity(72);
                for s in addr.segments().iter().rev() {
                    for c in format!("{:04x}", s).chars().rev() {
                        url.push(c);
                        url.push('.');
                    }
                }
                url.push_str("ip6.arpa");
                qtype = DnsType::PTR;
            }
        }
    }

    #[cfg(feature = "tls")]
    if connection_type == ConnectionType::TLS {
        if webpki::DnsNameRef::try_from_ascii_str(&nameserver).is_err() {
            err("Nameserver must be a valid hostname for TLS.");
        }
        if port.is_none() {
            port = Some(853);
        }
    }

    Args {
        nameserver,
        url,
        qtype,
        short,
        use_dnssec,
        connection_type,
        port: port.unwrap_or(53),
    }
}

fn print_help() {
    println!("Usage: \ttoluol [@nameserver] [domain] [q-type] [options] [flags]");
    println!("Where: \tq-type is a valid DNS QTYPE (AAAA, A, TXT, MX, SOA, ...)");
    println!("       \toptions is one or more of the following:");
    println!("       \t    -p | --port <port>     (use the given port number)");
    println!("       \t    -h | --help            (print this help message)");
    println!("       \t    -x                     (shortcut for reverse lookup)");
    println!("       \tflags is one or more of the following:");
    println!("       \t    +short  (only print answer section of response)");
    println!("       \t    +do     (fetch DNSSEC records)");
    println!("       \t    +tcp    (use TCP instead of UDP)");
    #[cfg(feature = "tls")]
    println!("       \t    +tls    (use DNS over TLS)");
    println!();
    println!("Note: order of the arguments does not matter.");
    println!(
        "If no arguments are specified, the default behaviour is `toluol @1.1.1.1 example.com A`."
    )
}

fn err(msg: impl AsRef<str>) {
    eprintln!("{}", msg.as_ref());
    process::exit(1);
}
